const dateFormat = require('../../../common/node_modules/dateformat');
/**
   * Convert a date String and return it's Unix timestamp.
   * @param {String} [format='epoch'] - Specify the format for the date
   * @param {String} [timestamp=''] - Specify 'now' to use the current time.
   * @category Transformers
   * @example
   * date                            // input 07-01-1985, output: '473904000' (The date/time epoch for January 7th 1985.)
   * date now                        // output: '1549468836' (The current date/time epoch.)
   * date nowms                      // output: '1549468836320' (The current date/time epoch in milliseconds.)
   * date ms                         // output: '320' (The current date/time epoch milliseconds. Number between 0 and 999.)
   * date 'yyyy-mm-dd hh:MM:ss'      // input 07-01-1985, output: '1985-01-07 00:00:00' (Formatted date/time epoch for January 7th 1985.)
   * date 'yyyy-mm-dd hh:MM:ss' now  // output: '2020-09-02 18:30:55' (Formatted current date/time.)
   * date epoch '1985-01-07'         // output: '473904000' (Formatted date/time epoch for January 7th 1985.)
   * date 'isoUtcDateTime'           // output: '2020-09-02T18:30:55Z' (ISO formatted current date/time.)
   * peek date::now                  // output: '1549468836', this time peeked from date scope function variable of which now, nowms and ms are available
   * data '$date::now'               // output: '1549468836', the date scope function variable can be read like any other variable
   */
exports.date = data => async function (p, timestampOrFormat, timestamp) {
  let format = 'epoch';
  let d;
  if (typeof timestampOrFormat === 'undefined' && typeof timestamp === 'undefined') {
    d = new Date(data).getTime();
  } else if (typeof timestampOrFormat !== 'undefined' && typeof timestamp === 'undefined') {
    if (timestampOrFormat === 'now') d = Math.round(Date.now() / 1E3);
    else if (timestampOrFormat === 'nowms') d = Date.now();
    else if (timestampOrFormat === 'ms') d = new Date().getMilliseconds();
    else if (!isNaN(Date.parse(timestampOrFormat))) d = new Date(timestampOrFormat).getTime();
    else {
      d = new Date(data).getTime();
      format = timestampOrFormat;
    }
  } else {
    d = timestamp === 'now'
      ? Date.now()
      : new Date(timestamp).getTime();
    format = timestampOrFormat;
  }

  if (isNaN(d)) return p.fail('date: could not parse date.');

  const dateOutput = format === 'epoch'
    ? d
    : dateFormat(d, format);
  return p.next(dateOutput);
};

exports.tests = {
  date: [
    'date now',
    "true '$>1500000000' 1 2",
    'done $OK',
    'fail'
  ],
  date1: [
    'date nowms',
    "true '$>1500000000000' 1 2",
    'done $OK',
    'fail'
  ],
  date2: [
    'date ms',
    'ship 1 1 2 2',
    'done $OK',
    'fail'
  ]
};
