const storageLib = require('../../../common/storage-fileBased');

/**
   * Retrieve a string value stored under key.
   * NOTE: When loading saved objects, pass them through jpar for object conversion.
   * @category Storage
   * @param {String} [key] - Key from which to retrieve data.
   * @example
   * load @success @notFound           // load from the key specified in the data stream, on success jump to @success, else jump to @notFound
   * load myFavoriteColor              // retrieve the value for myFavoriteColor
   * load myObject                     // retrieve the value for myObject
   * load myObject @success @notFound  // attempt to retrieve the value for myObject, jump to @success on success, or to @notFound on failure
   * jpar                              // often used in conjunction with load - converts a loaded JSON value to an object
   */
exports.load = data => async function (p, input, onSuccess, onFailure) {
  let key;
  if (input && isNaN(input)) {
    key = input;
  } else {
    key = data;
    onFailure = onSuccess;
    onSuccess = input;
  }
  if (typeof key !== 'string') return p.fail('load: Expecting key of type string');
  key = key.split(':')[0]; // remove any references to pubKey and signature
  if (typeof onSuccess === 'undefined') onSuccess = 1;
  if (isNaN(onSuccess)) return p.fail('load: Expecting jump for onSuccess');
  if (typeof onFailure !== 'undefined' && isNaN(onFailure)) return p.fail('load: Expecting jump for onFailure');

  const done = data => p.jump(onSuccess, data);
  const fail = error => {
    if (typeof onFailure === 'undefined') return p.fail(error);
    else return p.jump(onFailure, error);
  };
  return storageLib.load({key: encodeURIComponent(key), type: 'string'}, done, fail);
};

exports.tests = {
  load: [
    'data 123',
    'save testLoad',
    'load testLoad 1 3',
    "flow '123' 1 2",
    'done $OK',
    'fail'
  ]
};
