/**
   * Trim elements from left and or right side of a string or array.
   * @category Array/String
   * @param {String|Array} [leftOrBothTrim=' '] - The character(s) to trim from the ends of the string, or the left end if the second argument is specified.
   * @param {String|Array} [rightTrim=leftOrBothTrim] - The character(s) to trim from the right side of the string.
   * @param {Boolean} [reverseTrim=false] - Trim only the right side of the string. Useful for trimming only the rear side.
   * @example
   * trim                // input: "  hello world  ", output: 'hello world'
   * trim '_'            // input: "__hello_world___", output: 'hello_world'
   * trim '_' null true  // input: "__hello_world___", output: '__hello_world'
   * trim null '_' true  // input: "__hello_world___", output: '__hello_world'
   * trim ['_','-']      // input: "_-_hello-_world_-_-_", output: 'hello-_world'
   * trim ['_','-'] '_'  // input: "_-_hello-_world_-_-_", output: 'hello-_world_-_-'
   */
exports.trim = data => async function (p, leftOrBothTrim, rightTrim, rightOnly) {
  if (typeof data !== 'string' && !(data instanceof Array)) return p.fail('trim: Expected string or array!');

  if (typeof rightTrim !== 'undefined' && rightOnly === true) {
    rightTrim = leftOrBothTrim === null ? rightTrim : leftOrBothTrim;
    leftOrBothTrim = null;
  }

  if (typeof leftOrBothTrim === 'undefined') leftOrBothTrim = [' '];
  if (typeof rightTrim === 'undefined') rightTrim = leftOrBothTrim;

  let start = 0;
  let end;
  if (rightOnly !== true) for (start = 0; start < data.length && (data[start] === leftOrBothTrim || leftOrBothTrim.indexOf(data[start]) !== -1); ++start) {}
  for (end = data.length - 1; end >= 0 && (data[end] === rightTrim || (rightTrim !== null && rightTrim.hasOwnProperty('indexOf') && rightTrim.indexOf(data[end]) !== -1)); --end) {}

  const result = typeof data === 'string'
    ? data.substring(start, end + 1)
    : data.slice(start, end - start);

  return p.next(result);
};

exports.tests = {
  trim1: [
    "data ' hello '",
    'trim',
    'flow hello 1 2',
    'done $OK',
    'fail'
  ],
  trim2: [
    "data '_hello_'",
    'trim _',
    'flow hello 1 2',
    'done $OK',
    'fail'
  ],
  trim3: [
    "data '_hello-'",
    'trim _ -',
    'flow hello 1 2',
    'done $OK',
    'fail'
  ],
  trim4: [
    "data '-_hello_-'",
    'trim [_,-] -',
    'flow hello_ 1 2',
    'done $OK',
    'fail'
  ]
};
