const purgeTime = 86400; // 24 hours
const minimumRequiredBalance = 0.0001;
const math = require('../../lib/scheduler/math.js');

async function procCall(proc, command) {
  const result = await proc.call(command);
  return new Promise((resolve) => {
    if (typeof result === 'object' && result !== null && result.hasOwnProperty('data')) resolve(result.data);
    else resolve(null);
  });
}

async function findPair (proc) {
  let result = [];
  const fromBase = proc.command[1];
  const toSymbol = proc.command[2];
  const minimalAmount = proc.command[3];
  const accounts = proc.peek('local::accounts') || {};
  for (const accountID in accounts) {
    if(typeof accounts[accountID].pairs[`${fromBase}:${toSymbol}`] !== 'undefined') {
      let balanceSymbol = proc.peek(`supplyOverrideSymbols.${toSymbol}`) || toSymbol;
      let pair = accounts[accountID].pairs[`${fromBase}:${toSymbol}`];
      pair.accountID = accountID;
      pair.balance = proc.peek(`local::accounts[${accountID}].balances[${balanceSymbol}]`, 0);
      if (typeof pair.balance !== 'undefined') {
        let fueled = true;
        if (!balanceSymbol.startsWith('_') && balanceSymbol.includes('.')) {
          const fuelSymbol = balanceSymbol.split('.')[0];
          const fuelBalance = proc.peek(`local::accounts[${accountID}].balances[${fuelSymbol}]`, 0);
          const fuelFee = await procCall(proc, `${balanceSymbol}::fee`);
          if (!fuelFee || Number(fuelBalance) < (Number(fuelFee) * 2)) fueled = false;
        }
        if (fueled && Number(pair.balance) > Number(minimalAmount)) {
          const timeStamp = Math.floor( new Date() / 1E3 );
          proc.poke(`local::accounts[${accountID}].pairs[${fromBase}:${toSymbol}].active`,timeStamp); // update pair activity timestamp
          result.push(pair);
        }
      }
    }
  }
  return proc.done(result);
}

function removeStaleRecords (proc) {
  let result = [], timeStamp;
  const accountPurgeTime = Number( proc.command[1] || purgeTime );
  const accounts = proc.peek('local::accounts') || {};
  const timeNow = Math.floor( new Date() / 1E3 );
  const ledgers = proc.peek(`local::ledgers`, []);
  const minimumRequired = proc.peek(`conf::securityTransferFee`, 0) * 2;
  let mutation = false;
  for (const accountID in accounts) {
    // remove stale accounts
    // DEBUG: console.warn('> accountID: '+accountID);
    let removeAccount = false;
    if (accounts.hasOwnProperty(accountID) && typeof accounts[accountID] === 'object') {
      const timeStamp = !isNaN(accounts[accountID].created) ? Number(accounts[accountID].created) : 0;
      if ((timeStamp + accountPurgeTime) < timeNow) {
        if (accounts[accountID].hasOwnProperty('securityDepositBalances') || accounts[accountID].hasOwnProperty('securityLoanedBalances')) {
          let securityBalance = 0;
          if (accounts[accountID].hasOwnProperty('securityDepositBalances')) for (const symbol in accounts[accountID].securityDepositBalances) securityBalance += accounts[accountID].securityDepositBalances[symbol];
          if (accounts[accountID].hasOwnProperty('securityLoanedBalances')) for (const symbol in accounts[accountID].securityLoanedBalances) securityBalance += accounts[accountID].securityLoanedBalances[symbol];
          if (securityBalance < minimumRequired) removeAccount = true;
        } else removeAccount = true;
      }
      // remove stale pairs and/or old cached balances
      if (!removeAccount) {
        const pairs = accounts[accountID].pairs;
        for (const pair in pairs) {
          // DEBUG: console.warn('> '+pair);
          const fromBase = pair.split(':')[0];
          const toSymbol = pair.split(':')[1];
          // remove stale pair
          const timeStampActive = accounts[accountID].pairs[pair].hasOwnProperty('active') && !isNaN(accounts[accountID].pairs[pair].active) ? Number(accounts[accountID].pairs[pair].active) : 0;
          if ((timeStampActive + purgeTime) < timeNow && (ledgers.indexOf(fromBase) === -1  || ledgers.indexOf(toSymbol) === -1)) {
            mutation = true;
            delete accounts[accountID].pairs[pair];
            // DEBUG: console.warn('> remove pair: '+pair);
          }
        }
        const balanceSymbols = accounts[accountID].balances;
        for (const symbol in balanceSymbols) {
          // remove old cached balances
          const timeStampSync = accounts[accountID].hasOwnProperty('lastsync') && !isNaN(accounts[accountID].lastsync[symbol]) ? Number(accounts[accountID].lastsync[symbol]) : 0;
          const bidBalance = Number(accounts[accountID].balances[symbol]);
          // remove stale balance symbols
          if (ledgers.indexOf(symbol) === -1 || ((timeStampSync + purgeTime) < timeNow && bidBalance < minimumRequiredBalance)) {
            mutation = true;
            delete accounts[accountID].balances[symbol];
            delete accounts[accountID].feeModifiers[symbol];
            delete accounts[accountID].lastsync[symbol];
           // DEBUG: console.warn('> remove stale balance: '+symbol);
          }
        }
      }
    } else removeAccount = true;
    // save final mutations
    if (removeAccount) {
      mutation = true;
      delete accounts[accountID]; // delete stale account
      result.push(accountID);
    }
    // DEBUG: console.warn(JSON.stringify(removeAccount)+' '+JSON.stringify(mutation));
  }
  if (mutation) proc.poke(`local::accounts`, accounts);
  return proc.done(result);
}

function removeStalePairStats (proc) {
  let result = [], timeStamp;
  const statPurgeTime = Number( proc.command[1] || purgeTime );
  const pairstats = proc.peek('local::pairstats') || {};
  const timeNow = Math.floor( new Date() / 1E3 );
  const ledgers = proc.peek(`local::ledgers`, []);
  let mutation = false;
  for (const pair in pairstats) {
    // DEBUG: console.warn('> '+pair);
    const fromBase = pair.split(':')[0];
    const toSymbol = pair.split(':')[1];
    // remove stale pair
    const timeStampUpdated = pairstats[pair].hasOwnProperty('updated') && !isNaN(pairstats[pair].updated) ? Number(pairstats[pair].updated) : 0;
    if ((timeStampUpdated + statPurgeTime) < timeNow && (ledgers.indexOf(fromBase) === -1  || ledgers.indexOf(toSymbol) === -1)) {
      mutation = true;
      delete pairstats[pair];
      // DEBUG: console.warn('> remove pair: '+pair);
    }
  }
  //if (mutation) proc.poke(`local::pairstats`, accounts);
  return proc.done(result);
}

async function statSelectPairsNewOrIteratively (proc,allAccountedPairs) {
  const compileStatsMax = proc.command[1] || 3;
  let pairStats = proc.peek('local::pairstats') || {};
  if (typeof pairstats !== 'object') pairStats = {};
  let pairsSelect = [];
  // add new pairs to pairstats if necessary
  const flatPairStats = [];
  for (pair of allAccountedPairs) {
    if (flatPairStats.indexOf(pair) === -1) flatPairStats.push(pair);
    if (!pairStats.hasOwnProperty(pair)) {
      pairStats[pair] = {};
    }
  }
  // compile stats for known pairs
  let compilePairStatCount = proc.peek('local::compilePairStatCount', 0);
  for (let j = 0; j < compileStatsMax; j++) {
    //const flatPairStats = Object.keys(allAccountedPairKeys);
    const pair = flatPairStats[compilePairStatCount];
    pairsSelect.push(pair);
    if (compilePairStatCount >= flatPairStats.length) compilePairStatCount = 0;
    else compilePairStatCount++;
  }
  proc.poke('local::compilePairStatCount',compilePairStatCount);
  proc.done(pairsSelect);
}

function getVolumeAmount (command, data) {
  let result = 0;
  if (data.hasOwnProperty('deal')) {
    const deal = data.deal;
    if (deal.bid.symbol === command.toSymbol &&
        deal.ask.symbol === command.fromBase &&
        deal.ask.txid &&
        deal.bid.txid &&
        deal.pushtime &&
        !isNaN(deal.ask.amount)) result = deal.ask.amount;
  }
  return Number(result);
}

async function statGetVolumeAmount (proc, data) {
  const result = getVolumeAmount({toSymbol:proc.command[1],fromBase:proc.command[2]}, data);
  proc.done(result);
}

async function statGetPairStats (proc, data) {
  const functionRef = 'allocation-stats getPairStats';
  if (typeof data !== 'object'
      || (typeof data === 'object' && !(data.hasOwnProperty('value') && data.hasOwnProperty('meta'))))
        return proc.fail(500,`Expecting object with properties: pair, history  Instead: ${JSON.stringify(data)}`);

  const config = await procCall(proc, `allocation::shareConf`);
  if (config === null) return proc.fail(1,'Cannot load allocation configuration');
  const pair = data.value;
  if (typeof pair === 'undefined') return proc.fail(500,`Pair is undefined`);
  const symbols = pair.split(':');
  const fromBase = symbols[0];
  const toSymbol = symbols[1];
  const localAccounts = await procCall(proc, `allocation::getLocalAccounts`);
  if (localAccounts === null) return proc.fail(1,'Cannot load local accounts data');

  function statsCalculateCumulative (command, data) {
    const MAXVALUE = 999999999999999999;
    const action = command.action;
    const variable = command.variable;
    const pair = command.pair;
    const specifier = typeof command.specifier === 'undefined' ? pair.split(':')[1] : command.specifier;  
    let calcArray = [];
    for (const accountID in data) {
      let pushvalue;
      if (typeof data[accountID] !== 'undefined' && typeof data[accountID].pairs !== 'undefined' && data[accountID].pairs.hasOwnProperty(pair) && data[accountID].pairs[pair].hasOwnProperty(specifier) ) {
        pushvalue = data[accountID].pairs[pair][specifier];
        if (specifier === 'fee') {
          const feeSymbol = pair.split(':')[0];
          if (typeof feeSymbol !== 'undefined') {
            let feeModifier = proc.peek(`local::accounts[${accountID}].feeModifiers[${feeSymbol}]`, 1);
            feeModifier = isNaN(feeModifier) ? 1 : feeModifier;
            pushvalue = `${ Number(pushvalue.replace('%','')) * feeModifier }`;
          }
        } else pushvalue = data[accountID].pairs[pair][specifier];
      } else pushvalue = typeof data[accountID] !== 'undefined' && data[accountID].hasOwnProperty(variable) && data[accountID][variable].hasOwnProperty(specifier) ? data[accountID][variable][specifier] : 1;
      if (!isNaN(pushvalue) && pushvalue < MAXVALUE) calcArray.push(pushvalue); // filter out non-value entries and erroneously large balances
    }
    let result = null;
    switch (action) {
      case 'add':
      case 'average':
        result = 0;
        for (let i = 0; i < calcArray.length; i++ ) {
          result += Number(calcArray[i]);
        }
        if (action === 'average') result = result / calcArray.length;
        break;
      case 'min':
        result = MAXVALUE;
        for (let i = 0; i < calcArray.length; i++ ) {
          if (calcArray[i] !== 0 && result > calcArray[i]) result = Number(calcArray[i]);
        }
        if (result === MAXVALUE) result = 0;
        break;
      case 'max':
        result = 0;
        for (let i = 0; i < calcArray.length; i++ ) {
          if (result < calcArray[i]) result = Number(calcArray[i]);
        }
        break;
    }
    return result;
  }

  const totalSecurityBalances = statsCalculateCumulative({action:'max', variable:'securityDepositBalances', pair, specifier:config.securitySymbol}, localAccounts);
  const totalSecurityLoaned = statsCalculateCumulative({action:'max', variable:'securityLoanedBalances', pair, specifier:config.securitySymbol}, localAccounts);
  const totalSecurityLocked = statsCalculateCumulative({action:'max', variable:'securityLockedBalances', pair, specifier:config.securitySymbol}, localAccounts);
  const sufficiencySecurity = math.calc(`${totalSecurityBalances} + ${totalSecurityLoaned} - ${totalSecurityLocked}`).data;
  const sufficiencySecurityValuation = await procCall(proc, `valuations::rate/${config.securitySymbol}/${fromBase}/${sufficiencySecurity}`);
  if (sufficiencySecurityValuation === null) sufficiencySecurityValuation = 0;
  if (isNaN(sufficiencySecurityValuation) || sufficiencySecurityValuation <= 0) return proc.fail(500,`Could not determine valuation of risk sufficiency! -> ${sufficiencySecurityValuation}`+`  valuations::rate/${config.securitySymbol}/${fromBase}/${sufficiencySecurity}`);
  const sufficiencyRisk = math.calc(`${sufficiencySecurityValuation} / ${config.securityRiskFactor}`, functionRef).data;

  const sufficiencyBalance = statsCalculateCumulative({action:'max', variable:'balances', pair, specifier:toSymbol}, localAccounts);
  const sufficiencyBalanceValuation = await procCall(proc, `valuations::rate/${toSymbol}/${fromBase}/${sufficiencyBalance}`);
  if (sufficiencyBalanceValuation === null) sufficiencyBalanceValuation = 0;
  if (isNaN(sufficiencyBalanceValuation) || sufficiencyBalanceValuation <= 0) return proc.fail(500,`Could not determine valuation of pair sufficiency! -> ${sufficiencyBalanceValuation}`);
  const sufficiencyPair = math.calc(`${sufficiencyBalanceValuation} * (${config.liquidityPercentage} * 0.01)`, functionRef).data;

  let sufficiency = Number(sufficiencyPair < sufficiencyRisk ? sufficiencyPair : sufficiencyRisk); // choose smallest sufficiency

  const liquidityAddBalances = statsCalculateCumulative({action:'add', variable:'balances', pair, specifier:toSymbol}, localAccounts);
  let liquidity = isNaN(liquidityAddBalances) || liquidityAddBalances <= 0 ? 0 : await procCall(proc, `valuations::rate/${toSymbol}/${fromBase}/${liquidityAddBalances}`);
  if (isNaN(liquidity)) return proc.fail(500,'Could not determine valuation of liquidity amount');
  else liquidity = Number(liquidity);
  if (sufficiency>liquidity) sufficiency = liquidity;

  const minFee = statsCalculateCumulative({action:'min', variable:'pairs', pair, specifier:'fee'}, localAccounts);
  const maxFee = statsCalculateCumulative({action:'max', variable:'pairs', pair, specifier:'fee'}, localAccounts);

  // calculate price based on unit rate valuation * (1+feepercentage)
  const unitValuation = await procCall(proc, `valuations::rate/${toSymbol}/${fromBase}/1`);
  if (unitValuation === null) return proc.fail(1, 'Could not get unit valuation');
  const unitRate = isNaN(unitValuation) || unitValuation < 0 ? 0 : unitValuation;
  const bid = math.calc(`${unitRate} * (1 - (${minFee} * 0.01))`, functionRef).data;
  const askFeeMin = statsCalculateCumulative({action:'min', variable:'pairs', pair:`${toSymbol}:${fromBase}`, specifier:'fee'}, localAccounts);
  const askFee = isNaN(askFeeMin) || askFeeMin < 0 ? 0 : askFeeMin;
  const ask = askFee > 0 ? math.calc(`${unitRate} * (1 + (${askFee} * 0.01))`, functionRef).data : bid;

  // calculate 24 hour volume
  const readHistory = data.meta;
  //console.log( ' >>> ' + JSON.stringify(readHistory) + ' <<< ');
  let volume = 0, volumeHyValuation = 0;
  if (Array.isArray(readHistory)) {
    for (const entry of readHistory) {
      volume += getVolumeAmount({toSymbol,fromBase}, entry);
    }
    volumeHyValuation = await procCall(proc, `valuations::rate/${fromBase}/hy/${volume}`);
  }
  const volumeHY = isNaN(volumeHyValuation) || volumeHyValuation < 0 ? 0 : volumeHyValuation;

  // neatly present fee(s)
  const feeRange = minFee === maxFee ? `${minFee}%` : `R[${minFee}%,${maxFee}%]`;
  const timestamp = Math.round(Date.now() / 1E3);

  const result = {};
  result[pair] = {base:fromBase,ask,bid,volume,volumeHY,fee:feeRange,sufficiency,liquidity,updated:timestamp};
  proc.done(result);
}

// exports
exports.findPair = findPair;
exports.removeStaleRecords = removeStaleRecords;
exports.removeStalePairStats = removeStalePairStats;
exports.getPairStats = statGetPairStats;
exports.statSelectPairsNewOrIteratively = statSelectPairsNewOrIteratively;
exports.statGetVolumeAmount = statGetVolumeAmount;
