// cache.js -> handles cache clearing
//
// (c)2018 internet of coins project - Rouke Pouw
//

const sequential = require('./util/sequential');
const conf = require('./conf/conf');
const cached = {}; // cached responses
let cacheInterval;

async function init (callbackArray) {
  // periodically clean the request cache
  cacheInterval = setInterval(function () {
    const maxentries = conf.get('cache.maxentries');
    const cachedarray = Object.keys(cached);
    if (cachedarray.length > maxentries) {
      for (let i = 0; i < cachedarray.length - maxentries; i++) {
        delete cached[cachedarray[i]];
      }
    }
  }, conf.get('cache.cleanupinterval'));
  // this sequential call is put here async by purpose, the cache clearer needs te be started, is does not need to be run a first time before continuing
  return sequential.next(callbackArray);
}

function isRunning () {
  return !!cacheInterval;
}

// pause the APIqueue
async function pause (cbArr) {
  if (isRunning()) {
    global.hybrixd.logger(['info', 'cache'], 'Cache clean up paused');
  }
  clearInterval(cacheInterval);
  cacheInterval = null;
  return sequential.next(cbArr);
}

// resume the APIqueue
async function resume (cbArr) {
  if (!isRunning()) {
    global.hybrixd.logger(['info', 'cache'], 'Cache clean up started');
    init(cbArr);
  } else {
    return sequential.next(cbArr);
  }
}

async function clear (callbackArray) {
  for (let index in cached) { if (cached.hasOwnProperty(index)) { delete cached[index]; } }
  return sequential.next(callbackArray);
}

async function add (index, data) {
  if (!(cached.hasOwnProperty(index) && cached[index][1] === data)) cached[index] = [Date.now(), data];
}

function get (index, millisecs) {
  //const freshThreshold = conf.get('cache.freshthreshold');
  //if (millisecs < freshThreshold) { millisecs = freshThreshold; }
  return cached.hasOwnProperty(index) // cache exists for this request
    ? {
      data: cached[index][1],
      fresh: cached[index][0] > (Date.now() - Number(millisecs)) } // the time does not exceed max allowed delta, it is still fresh
    : null;
}

// export every function
exports.init = init;
exports.add = add;
exports.pause = pause;
exports.resume = resume;
exports.get = get;
exports.clear = clear;
