// (C) 2025 hybrix / Joachim de Koning
// hybrixd module - assetfunctions/module.js
// Helper module for exotic functions pertaining to assets

// required libraries in this context
const baseCode = require('../../common/basecode');
const sha256 = require('../../common/node_modules/js-sha256');

const bchaddr = require('bchaddrjs');
function cashaddrToLegacy(proc) {
  const address = proc.command[1];
  const result = bchaddr.toLegacyAddress(address);
  return proc.done(result);
}

function factorialFindTransactionByProof(proc, data) {
  const symbol = proc.command[1];
  const source = proc.command[2];
  const proof = proc.command[3];
  const timestamp = proc.command[4];
  const gracetime = proc.command[5];
  let transactions = [];
  if (data instanceof Array) {
    for (tx of data) {
      const txTimestamp = Number(tx.fcctime);
      const txProof = factorialMakeProof(symbol, source, tx);
      if ((txTimestamp >= Number(timestamp) && txTimestamp < (Number(timestamp) + Number(gracetime))) && txProof === proof) transactions.push(tx);
    }
  } else proc.fail(1,'Expecting array as input for findFactorialTransactionByProof');
  if (transactions.length === 1) proc.done(transactions[0]);
  else if (transactions.length > 1) proc.fail('Multiple transactions listed under same proof!');
  else proc.done(null);
}

function factorialHistoryFormat(proc, data) {
  const symbol = proc.command[1];
  const source = proc.command[2];
  const epochOffset = proc.peek('proc::epochOffset', 0);
  let transactionHashes = [];
  if (data instanceof Array) {
    for (tx of data) {
      const txidSource = tx.balance === 'credit' ? source : tx.from; 
      const txTimestamp = baseCode.recode('dec', 'hex', String(Number(tx.fcctime) - Number(epochOffset))).slice(0,-2).toUpperCase();
      const txProof = factorialMakeProof(symbol, source, tx);
      if (txProof) transactionHashes.push(`${txProof}${txidSource}${txTimestamp}`);
    }
  }
  proc.done(transactionHashes);
}

function factorialMakeProof(symbol, source, tx) {
    let target, txAmount, txFee;
    if (tx.hasOwnProperty('to') && tx.to instanceof Array && tx.to[0].hasOwnProperty('wallet')) {
      target = tx.to[0].wallet;
      txAmount = tx.to[0].amount;
    } else if (tx.hasOwnProperty('from')) {
      target = source;
      source = tx.from;
      txAmount = tx.amount;
    }
  if (target) return sha256.sha256(`${symbol} ${source} ${txAmount} ${target}`).substr(0,16).toUpperCase();
  else return false;
}

// exports
exports.cashaddrToLegacy = cashaddrToLegacy;
exports.factorialFindTransactionByProof = factorialFindTransactionByProof;
exports.factorialHistoryFormat = factorialHistoryFormat;
