let privateToken; // used to upgrade session to root

const fetchQuery = (query) => {
  return new Promise((resolve, reject) => {
    request(query, resolve, reject);
  });
};

function setCookie (fields, exdays) {
  const d = new Date();
  d.setTime(d.getTime() + (exdays * 24 * 60 * 60 * 1000));
  const expires = 'expires=' + d.toUTCString();

  for (const key in fields) {
    document.cookie = key + '=' + fields[key] + ';' + expires + ';path=/';
  }
}

function getCookie () {
  if (document.cookie === '') { return null; }
  const fields = document.cookie.split(';');
  const r = {};
  for (let i = 0; i < fields.length; i++) {
    const values = fields[i].split('=');
    r[values[0].trim()] = values[1].trim();
  }
  return r;
}

const cleanEmpty = obj => {
  if (Array.isArray(obj)) {
    return obj
      .map(v => (v && typeof v === 'object') ? cleanEmpty(v) : v)
      .filter(v => !(JSON.stringify(v) === '{}'));
  } else {
    return Object.entries(obj)
      .map(([k, v]) => [k, v && typeof v === 'object' ? cleanEmpty(v) : v])
      .reduce((a, [k, v]) => (JSON.stringify(v) === '{}' ? a : {...a, [k]: v}), {});
  }
};

function findNodes (obj, property) {
  const p = property || 'leaf';

  return Object
    .keys(obj)
    .reduce((obj_, key) => {
      const value = obj[key];
      if (key === '_hidden' && value === true) for (const x in obj) delete obj[x]; // hide hidden _ui points
      if (typeof value === 'object' && value !== null) {
        obj_[key] = findNodes(value, p);
      } else if (key === p) {
        obj_[p] = obj[p];
      }
      return obj_;
    }, {});
}

const paths = (obj, prop, str) => {
  return Object.keys(obj)
    .reduce((obj_, k) => {
      const value = obj[k];
      const valLength = Object.keys(value).length;

      if (k === prop) {
        obj_[k] = str;
      } else if (typeof value === 'object' && value !== null) {
        if (Object.keys(obj).length === 1 && valLength === 1 && value[prop] !== undefined) {
          obj_[k] = {[prop]: str + '/' + value[prop]};
        } else if (valLength === 1 && value[prop] === undefined) {
          obj_ = {...obj_, ...paths(value, prop, `${str}/${k}`)};
        } else {
          obj_[k] = paths(value, prop, str + '/' + k);
        }
      }

      return obj_;
    }, {});
};

const DEFAULT_RETRIES = 60;
const FIRST_RETRY_DELAY = 100;
const DEFAULT_RETRY_DELAY = 500;

function determineHost () {
  return window.location.origin + (window.location.pathname.startsWith('/api') ? '/api' : '');
}

function request (url, dataCallback, errorCallback, progressCallback, debug = false, retries) {
  let host;
  if (!url.includes('://')) { // '/a/dummy/details'
    host = determineHost();
  } else if (url.includes('/api')) { // 'https://myhost.com/api/a/dummy/details'
    [host, url] = url.split('/api');
    host = host + '/api';
  } else { // 'https://myhost.com/a/dummy/details'
    const [protocol, x] = url.split('://'); // ['https','myhost.com/a/dummy/details']
    const path = x.split('/'); // ['myhost.com','a','dummy','details']
    host = protocol + '://' + path[0]; // 'https://myhost.com'
    url = '/' + path.slice(1).join('/'); // '/a/dummy/details'
  }

  if (typeof retries === 'undefined') retries = DEFAULT_RETRIES;

  const xhr = new XMLHttpRequest();
  //console.log(' GET: '+host +' '+ url);
  xhr.open('GET', host + url, true);
  if (privateToken) xhr.setRequestHeader('private-token', privateToken);

  xhr.onreadystatechange = (e) => {
    if (xhr.readyState === 4) {
      let result;

      if (xhr.getResponseHeader('Content-Type') === 'text/html') {
        dataCallback({html: xhr.responseText});
        return;
      }

      try { // Catch bad parse
        result = JSON.parse(xhr.responseText);
      } catch (e) {
        errorCallback(e);
        return;
      }
      if (typeof result !== 'object' || result === null) {
        errorCallback('Invalid response: ' + xhr.responseText);
        return;
      }

      if (result.error !== 0) {
        let help;
        if (result.help) {
          help = result.help.replace(/`(.*?)`/g, (a, b) =>
            (`<a onclick="document.getElementById('request').value='${b}';if(!'${b}'.includes('$')){go();}">${b}</a>`));
        } else {
          help = result.data;
        }

        errorCallback(help);
        return;
      }

      if (result.hasOwnProperty('id') && result.id === 'id') { // requires follow up
        request(host + '/p/' + (debug ? 'debug/' : '') + result.data, dataCallback, errorCallback, progressCallback, debug);
      } else if (result.stopped !== null) { // done
        dataCallback(result.data);
      } else { // not yet finished
        if (typeof progressCallback === 'function') {
          progressCallback(1.0 - retries / DEFAULT_RETRIES, result.progress);
        }
        if (retries === 0) { // Max retries exceeded
          errorCallback('Timeout');
        } else { // retry
          setTimeout(() => {
            request(host + url, dataCallback, errorCallback, progressCallback, debug, retries - 1);
          }, retries === DEFAULT_RETRIES ? FIRST_RETRY_DELAY : DEFAULT_RETRY_DELAY);
        }
      }
    }
  };
  xhr.send();
}

window.request = request;

function menuWithEndpoints (obj, prop) {
  return paths(
    cleanEmpty(
      findNodes(obj, prop)),
    prop, '');
}

function logout () {
  privateToken = undefined;
  const navLogin = document.getElementById('login');
  navLogin.innerHTML = 'login';
  navLogin.onclick = showLogin;
}

function showLogin () {
  const DIV = document.createElement('DIV');
  DIV.classList.add('login');
  const INPUT = document.createElement('INPUT');
  DIV.appendChild(INPUT);

  const ALogin = document.createElement('A');
  ALogin.innerHTML = 'login';
  ALogin.onclick = () => {
    setCookie({'private-token': INPUT.value});
    privateToken = INPUT.value;
    document.body.removeChild(DIV);
    const navLogin = document.getElementById('login');
    navLogin.innerHTML = 'logout';
    navLogin.onclick = logout;
  };
  DIV.appendChild(ALogin);
  const ACancel = document.createElement('A');
  ACancel.innerHTML = 'cancel';
  ACancel.onclick = () => document.body.removeChild(DIV);
  DIV.appendChild(ACancel);

  document.body.appendChild(DIV);
}

function addUIlinks (items) {
  const NAV = document.querySelector('.navModuleHeader > .navModuleNavigation');
  Object.keys(items).reverse().forEach(function (key) {
    const item = items[key];
    if (item.hasOwnProperty('_ui')) {
      const a = document.createElement('A');
      a.innerHTML = key;
      a.href = determineHost() + item._ui;
      NAV.insertAdjacentElement('afterbegin', a);
    } else addUIlinks(item);
  });
}

function renderMenu (items) {
  const hasRootAccess = items.hasOwnProperty('debug');

  addUIlinks(items);

  if (!hasRootAccess || privateToken) {
    const a = document.createElement('A');
    a.id = 'login';
    if (privateToken) {
      a.innerHTML = 'logout';
      a.onclick = logout;
    } else {
      a.innerHTML = 'login';
      a.onclick = showLogin;
    }
    document.querySelector('.navModuleHeader > .navModuleNavigation').appendChild(a);
  }
}

function renderHeader () {
  const header = `<header class="navModuleHeader">
  <div class="logo navModuleLogo">
    <svg viewBox="0 0 120 49" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
        <defs>
          <radialGradient cx="74.8280232%" cy="94.3295877%" fx="74.8280232%" fy="94.3295877%" r="55.5402692%" gradientTransform="translate(0.748280,0.943296),scale(0.405168,1.000000),rotate(-103.463924),scale(1.000000,3.068031),translate(-0.748280,-0.943296)" id="radialGradient-1">
              <stop stop-color="#69E1CD" offset="0%"></stop>
              <stop stop-color="#69E1CE" stop-opacity="0" offset="100%"></stop>
          </radialGradient>
          <radialGradient cx="2.43573207%" cy="10.0784095%" fx="2.43573207%" fy="10.0784095%" r="177.087164%" gradientTransform="translate(0.024357,0.100784),scale(0.405168,1.000000),rotate(-140.508079),scale(1.000000,1.714322),translate(-0.024357,-0.100784)" id="radialGradient-2">
              <stop stop-color="#EAA05B" offset="0%"></stop>
              <stop stop-color="#EAA05B" stop-opacity="0.914968297" offset="6.17548772%"></stop>
              <stop stop-color="#EAA05B" stop-opacity="0" offset="100%"></stop>
              <stop stop-color="#EAA05B" stop-opacity="0" offset="100%"></stop>
          </radialGradient>
          <radialGradient cx="93.7433601%" cy="0%" fx="93.7433601%" fy="0%" r="230.126538%" gradientTransform="translate(0.937434,0.000000),scale(0.405168,1.000000),rotate(145.390210),scale(1.000000,1.466156),translate(-0.937434,-0.000000)" id="radialGradient-3">
              <stop stop-color="#F252BC" offset="0%"></stop>
              <stop stop-color="#F252BC" stop-opacity="0.891502491" offset="31.6104241%"></stop>
              <stop stop-color="#F251BD" stop-opacity="0" offset="100%"></stop>
          </radialGradient>
          <radialGradient cx="1.95055786%" cy="17.1506499%" fx="1.95055786%" fy="17.1506499%" r="92.519592%" gradientTransform="translate(0.019506,0.171506),scale(0.405168,1.000000),rotate(-116.284375),scale(1.000000,1.779837),translate(-0.019506,-0.171506)" id="radialGradient-4">
              <stop stop-color="#EAA05B" offset="0%"></stop>
              <stop stop-color="#EAA05B" stop-opacity="0.914968297" offset="6.17548772%"></stop>
              <stop stop-color="#EAA05B" stop-opacity="0" offset="100%"></stop>
              <stop stop-color="#EAA05B" stop-opacity="0" offset="100%"></stop>
          </radialGradient>
          <radialGradient cx="60.5274555%" cy="111.294064%" fx="60.5274555%" fy="111.294064%" r="69.1214675%" gradientTransform="translate(0.605275,1.112941),scale(0.405168,1.000000),rotate(-96.719363),scale(1.000000,1.766288),translate(-0.605275,-1.112941)" id="radialGradient-5">
              <stop stop-color="#69E1CD" offset="0%"></stop>
              <stop stop-color="#69E1CE" stop-opacity="0" offset="100%"></stop>
              <stop stop-color="#69E1CE" stop-opacity="0" offset="100%"></stop>
          </radialGradient>
          <radialGradient cx="21.7131541%" cy="100%" fx="21.7131541%" fy="100%" r="69.1214675%" gradientTransform="translate(0.217132,1.000000),scale(0.405168,1.000000),rotate(-96.719363),scale(1.000000,1.766288),translate(-0.217132,-1.000000)" id="radialGradient-6">
              <stop stop-color="#69E1CD" stop-opacity="0.120669158" offset="0%"></stop>
              <stop stop-color="#69E1CE" stop-opacity="0" offset="100%"></stop>
              <stop stop-color="#69E1CE" stop-opacity="0" offset="100%"></stop>
          </radialGradient>
          <radialGradient cx="56.5147907%" cy="124.171979%" fx="56.5147907%" fy="124.171979%" r="55.1388182%" gradientTransform="translate(0.565148,1.241720),scale(0.405168,1.000000),rotate(-96.149962),scale(1.000000,3.608864),translate(-0.565148,-1.241720)" id="radialGradient-7">
              <stop stop-color="#69E1CD" offset="0%"></stop>
              <stop stop-color="#69E1CE" stop-opacity="0" offset="100%"></stop>
          </radialGradient>
          <radialGradient cx="85.0772955%" cy="73.9650222%" fx="85.0772955%" fy="73.9650222%" r="37.8033745%" gradientTransform="translate(0.850773,0.739650),scale(0.405168,1.000000),rotate(-90.000000),scale(1.000000,2.958850),translate(-0.850773,-0.739650)" id="radialGradient-8">
              <stop stop-color="#60BDC8" offset="0%"></stop>
              <stop stop-color="#60BCC8" stop-opacity="0" offset="100%"></stop>
          </radialGradient>
          <radialGradient cx="29.2186775%" cy="111.848772%" fx="29.2186775%" fy="111.848772%" r="42.4086048%" gradientTransform="translate(0.292187,1.118488),scale(0.405168,1.000000),rotate(-90.000000),scale(1.000000,2.958850),translate(-0.292187,-1.118488)" id="radialGradient-9">
              <stop stop-color="#7BD2B8" offset="0%"></stop>
              <stop stop-color="#7DD1B6" stop-opacity="0" offset="100%"></stop>
          </radialGradient>
        </defs>
        <g stroke="none" stroke-width="1" fill="none" fill-rule="evenodd">
          <g fill-rule="nonzero">
              <path d="M52.5400582,12.834102 C58.439645,12.834102 63.2395014,17.868213 63.2395014,24.0547741 C63.2395014,30.2413351 58.439645,35.274573 52.5400582,35.274573 C50.8469303,35.2746382 49.1802607,34.8546496 47.6895391,34.0522776 L47.6895391,35.1741702 L41.1444392,35.1741702 L41.1444392,0 L47.6895391,0 L47.6895391,14.0563974 C49.1802607,13.2540254 50.8469303,12.8340368 52.5400582,12.834102 Z M52.5400582,28.4122573 C54.8329832,28.4122573 56.697022,26.4565846 56.697022,24.0530279 C56.697022,21.6494713 54.8321097,19.6937986 52.5400582,19.6937986 C50.2480066,19.6937986 48.3830943,21.6485982 48.3830943,24.0530279 C48.3830943,26.4574576 50.2480066,28.4122573 52.5400582,28.4122573 Z M72.6482086,12.6811259 L72.6482086,13.8872144 C74.1374669,13.109131 75.790596,12.6960273 77.471158,12.681998 L77.471158,19.5653356 C74.8549807,19.6185326 72.7355018,21.8737348 72.6482086,24.6766041 L72.6482086,35.3299491 L66.0636823,35.3299491 L66.0636823,12.6811259 L72.6482086,12.6811259 Z M38.1235798,12.6257498 L38.1218341,24.9499075 L38.15413,24.9499075 C38.1445285,25.0493663 38.1340542,25.148825 38.1218341,25.2474113 L38.1218341,48.6202121 L31.2183678,48.6202121 L31.2183678,34.2946657 C29.8617641,34.8533559 28.4085877,35.1402952 26.9413435,35.1391924 C21.0905488,35.1391924 16.2679857,30.658315 15.7268112,24.9499075 L22.6974879,24.9499075 C23.1855033,26.8844709 24.9212773,28.2438906 26.9173121,28.2547803 C28.913347,28.2656701 30.6638644,26.9252704 31.1729789,24.9961471 L31.1197343,24.9961471 L31.1101329,24.9533973 L31.2148763,24.5285166 L31.2201135,24.7169647 L31.2201135,12.6257498 L38.1235798,12.6257498 Z M81.0152282,12.5149977 L87.7157358,12.5149977 L87.7157358,35.1638208 L81.0152282,35.1638208 L81.0152282,12.5149977 Z M11.2180264,12.5716929 C17.0453911,12.5716929 21.8552379,17.0154371 22.4273188,22.6910326 L15.4400709,22.6910326 C14.9217769,20.784737 13.1903773,19.4616421 11.2140961,19.4616421 C9.23781493,19.4616421 7.50641527,20.784737 6.98812126,22.6910326 L6.91650197,22.981753 L6.91650197,35.3299491 L0.0165947137,35.3299491 L0.0165947137,22.6910326 L0,22.6910326 C0.00436702991,22.6412697 0.0113542778,22.5923797 0.0165947137,22.5434898 L0.0165947137,0 L6.91650197,0 L6.91650197,13.4246473 C8.28018268,12.8600325 9.74198978,12.570169 11.2180264,12.5716929 Z M108.306903,19.4569596 L108.315643,19.4569596 C105.910484,19.4908081 103.978508,21.4469456 103.978508,23.8483362 C103.978508,26.2497267 105.910484,28.2058643 108.315643,28.2397127 L108.315643,35.125426 C105.06582,35.1264609 101.973734,33.7269236 99.8328939,31.2859711 C97.6907196,33.7498118 94.5831631,35.1645857 91.315182,35.1638208 L91.315182,28.2772346 C93.7442786,28.2772346 95.7134505,26.3113437 95.7134505,23.8862944 C95.7134505,21.4612451 93.7442786,19.4953542 91.315182,19.4953542 L91.315182,12.6087683 C94.5633322,12.6096753 97.6533072,14.008978 99.7935612,16.4482232 C101.934735,13.9855327 105.040447,12.5708729 108.306903,12.5703737 L108.306903,19.4569596 Z M115.5976,35.6068296 C113.166222,35.6068296 111.1952,33.6358082 111.1952,31.20443 C111.1952,28.7730518 113.166222,26.8020304 115.5976,26.8020304 C118.028978,26.8020304 120,28.7730518 120,31.20443 C120,33.6358082 118.028978,35.6068296 115.5976,35.6068296 Z" fill="#FFFFFF"></path>
              <path d="M52.5400582,12.834102 C58.439645,12.834102 63.2395014,17.868213 63.2395014,24.0547741 C63.2395014,30.2413351 58.439645,35.274573 52.5400582,35.274573 C50.8469303,35.2746382 49.1802607,34.8546496 47.6895391,34.0522776 L47.6895391,35.1741702 L41.1444392,35.1741702 L41.1444392,0 L47.6895391,0 L47.6895391,14.0563974 C49.1802607,13.2540254 50.8469303,12.8340368 52.5400582,12.834102 Z M52.5400582,28.4122573 C54.8329832,28.4122573 56.697022,26.4565846 56.697022,24.0530279 C56.697022,21.6494713 54.8321097,19.6937986 52.5400582,19.6937986 C50.2480066,19.6937986 48.3830943,21.6485982 48.3830943,24.0530279 C48.3830943,26.4574576 50.2480066,28.4122573 52.5400582,28.4122573 Z M72.6482086,12.6811259 L72.6482086,13.8872144 C74.1374669,13.109131 75.790596,12.6960273 77.471158,12.681998 L77.471158,19.5653356 C74.8549807,19.6185326 72.7355018,21.8737348 72.6482086,24.6766041 L72.6482086,35.3299491 L66.0636823,35.3299491 L66.0636823,12.6811259 L72.6482086,12.6811259 Z M38.1235798,12.6257498 L38.1218341,24.9499075 L38.15413,24.9499075 C38.1445285,25.0493663 38.1340542,25.148825 38.1218341,25.2474113 L38.1218341,48.6202121 L31.2183678,48.6202121 L31.2183678,34.2946657 C29.8617641,34.8533559 28.4085877,35.1402952 26.9413435,35.1391924 C21.0905488,35.1391924 16.2679857,30.658315 15.7268112,24.9499075 L22.6974879,24.9499075 C23.1855033,26.8844709 24.9212773,28.2438906 26.9173121,28.2547803 C28.913347,28.2656701 30.6638644,26.9252704 31.1729789,24.9961471 L31.1197343,24.9961471 L31.1101329,24.9533973 L31.2148763,24.5285166 L31.2201135,24.7169647 L31.2201135,12.6257498 L38.1235798,12.6257498 Z M81.0152282,12.5149977 L87.7157358,12.5149977 L87.7157358,35.1638208 L81.0152282,35.1638208 L81.0152282,12.5149977 Z M11.2180264,12.5716929 C17.0453911,12.5716929 21.8552379,17.0154371 22.4273188,22.6910326 L15.4400709,22.6910326 C14.9217769,20.784737 13.1903773,19.4616421 11.2140961,19.4616421 C9.23781493,19.4616421 7.50641527,20.784737 6.98812126,22.6910326 L6.91650197,22.981753 L6.91650197,35.3299491 L0.0165947137,35.3299491 L0.0165947137,22.6910326 L0,22.6910326 C0.00436702991,22.6412697 0.0113542778,22.5923797 0.0165947137,22.5434898 L0.0165947137,0 L6.91650197,0 L6.91650197,13.4246473 C8.28018268,12.8600325 9.74198978,12.570169 11.2180264,12.5716929 Z M108.306903,19.4569596 L108.315643,19.4569596 C105.910484,19.4908081 103.978508,21.4469456 103.978508,23.8483362 C103.978508,26.2497267 105.910484,28.2058643 108.315643,28.2397127 L108.315643,35.125426 C105.06582,35.1264609 101.973734,33.7269236 99.8328939,31.2859711 C97.6907196,33.7498118 94.5831631,35.1645857 91.315182,35.1638208 L91.315182,28.2772346 C93.7442786,28.2772346 95.7134505,26.3113437 95.7134505,23.8862944 C95.7134505,21.4612451 93.7442786,19.4953542 91.315182,19.4953542 L91.315182,12.6087683 C94.5633322,12.6096753 97.6533072,14.008978 99.7935612,16.4482232 C101.934735,13.9855327 105.040447,12.5708729 108.306903,12.5703737 L108.306903,19.4569596 Z M115.5976,35.6068296 C113.166222,35.6068296 111.1952,33.6358082 111.1952,31.20443 C111.1952,28.7730518 113.166222,26.8020304 115.5976,26.8020304 C118.028978,26.8020304 120,28.7730518 120,31.20443 C120,33.6358082 118.028978,35.6068296 115.5976,35.6068296 Z" fill-opacity="0.500990716" fill="#8F7BB6"></path>
              <path d="M52.5400582,12.834102 C58.439645,12.834102 63.2395014,17.868213 63.2395014,24.0547741 C63.2395014,30.2413351 58.439645,35.274573 52.5400582,35.274573 C50.8469303,35.2746382 49.1802607,34.8546496 47.6895391,34.0522776 L47.6895391,35.1741702 L41.1444392,35.1741702 L41.1444392,0 L47.6895391,0 L47.6895391,14.0563974 C49.1802607,13.2540254 50.8469303,12.8340368 52.5400582,12.834102 Z M52.5400582,28.4122573 C54.8329832,28.4122573 56.697022,26.4565846 56.697022,24.0530279 C56.697022,21.6494713 54.8321097,19.6937986 52.5400582,19.6937986 C50.2480066,19.6937986 48.3830943,21.6485982 48.3830943,24.0530279 C48.3830943,26.4574576 50.2480066,28.4122573 52.5400582,28.4122573 Z M72.6482086,12.6811259 L72.6482086,13.8872144 C74.1374669,13.109131 75.790596,12.6960273 77.471158,12.681998 L77.471158,19.5653356 C74.8549807,19.6185326 72.7355018,21.8737348 72.6482086,24.6766041 L72.6482086,35.3299491 L66.0636823,35.3299491 L66.0636823,12.6811259 L72.6482086,12.6811259 Z M38.1235798,12.6257498 L38.1218341,24.9499075 L38.15413,24.9499075 C38.1445285,25.0493663 38.1340542,25.148825 38.1218341,25.2474113 L38.1218341,48.6202121 L31.2183678,48.6202121 L31.2183678,34.2946657 C29.8617641,34.8533559 28.4085877,35.1402952 26.9413435,35.1391924 C21.0905488,35.1391924 16.2679857,30.658315 15.7268112,24.9499075 L22.6974879,24.9499075 C23.1855033,26.8844709 24.9212773,28.2438906 26.9173121,28.2547803 C28.913347,28.2656701 30.6638644,26.9252704 31.1729789,24.9961471 L31.1197343,24.9961471 L31.1101329,24.9533973 L31.2148763,24.5285166 L31.2201135,24.7169647 L31.2201135,12.6257498 L38.1235798,12.6257498 Z M81.0152282,12.5149977 L87.7157358,12.5149977 L87.7157358,35.1638208 L81.0152282,35.1638208 L81.0152282,12.5149977 Z M11.2180264,12.5716929 C17.0453911,12.5716929 21.8552379,17.0154371 22.4273188,22.6910326 L15.4400709,22.6910326 C14.9217769,20.784737 13.1903773,19.4616421 11.2140961,19.4616421 C9.23781493,19.4616421 7.50641527,20.784737 6.98812126,22.6910326 L6.91650197,22.981753 L6.91650197,35.3299491 L0.0165947137,35.3299491 L0.0165947137,22.6910326 L0,22.6910326 C0.00436702991,22.6412697 0.0113542778,22.5923797 0.0165947137,22.5434898 L0.0165947137,0 L6.91650197,0 L6.91650197,13.4246473 C8.28018268,12.8600325 9.74198978,12.570169 11.2180264,12.5716929 Z M108.306903,19.4569596 L108.315643,19.4569596 C105.910484,19.4908081 103.978508,21.4469456 103.978508,23.8483362 C103.978508,26.2497267 105.910484,28.2058643 108.315643,28.2397127 L108.315643,35.125426 C105.06582,35.1264609 101.973734,33.7269236 99.8328939,31.2859711 C97.6907196,33.7498118 94.5831631,35.1645857 91.315182,35.1638208 L91.315182,28.2772346 C93.7442786,28.2772346 95.7134505,26.3113437 95.7134505,23.8862944 C95.7134505,21.4612451 93.7442786,19.4953542 91.315182,19.4953542 L91.315182,12.6087683 C94.5633322,12.6096753 97.6533072,14.008978 99.7935612,16.4482232 C101.934735,13.9855327 105.040447,12.5708729 108.306903,12.5703737 L108.306903,19.4569596 Z M115.5976,35.6068296 C113.166222,35.6068296 111.1952,33.6358082 111.1952,31.20443 C111.1952,28.7730518 113.166222,26.8020304 115.5976,26.8020304 C118.028978,26.8020304 120,28.7730518 120,31.20443 C120,33.6358082 118.028978,35.6068296 115.5976,35.6068296 Z" fill="url(#radialGradient-1)"></path>
              <path d="M52.5400582,12.834102 C58.439645,12.834102 63.2395014,17.868213 63.2395014,24.0547741 C63.2395014,30.2413351 58.439645,35.274573 52.5400582,35.274573 C50.8469303,35.2746382 49.1802607,34.8546496 47.6895391,34.0522776 L47.6895391,35.1741702 L41.1444392,35.1741702 L41.1444392,0 L47.6895391,0 L47.6895391,14.0563974 C49.1802607,13.2540254 50.8469303,12.8340368 52.5400582,12.834102 Z M52.5400582,28.4122573 C54.8329832,28.4122573 56.697022,26.4565846 56.697022,24.0530279 C56.697022,21.6494713 54.8321097,19.6937986 52.5400582,19.6937986 C50.2480066,19.6937986 48.3830943,21.6485982 48.3830943,24.0530279 C48.3830943,26.4574576 50.2480066,28.4122573 52.5400582,28.4122573 Z M72.6482086,12.6811259 L72.6482086,13.8872144 C74.1374669,13.109131 75.790596,12.6960273 77.471158,12.681998 L77.471158,19.5653356 C74.8549807,19.6185326 72.7355018,21.8737348 72.6482086,24.6766041 L72.6482086,35.3299491 L66.0636823,35.3299491 L66.0636823,12.6811259 L72.6482086,12.6811259 Z M38.1235798,12.6257498 L38.1218341,24.9499075 L38.15413,24.9499075 C38.1445285,25.0493663 38.1340542,25.148825 38.1218341,25.2474113 L38.1218341,48.6202121 L31.2183678,48.6202121 L31.2183678,34.2946657 C29.8617641,34.8533559 28.4085877,35.1402952 26.9413435,35.1391924 C21.0905488,35.1391924 16.2679857,30.658315 15.7268112,24.9499075 L22.6974879,24.9499075 C23.1855033,26.8844709 24.9212773,28.2438906 26.9173121,28.2547803 C28.913347,28.2656701 30.6638644,26.9252704 31.1729789,24.9961471 L31.1197343,24.9961471 L31.1101329,24.9533973 L31.2148763,24.5285166 L31.2201135,24.7169647 L31.2201135,12.6257498 L38.1235798,12.6257498 Z M81.0152282,12.5149977 L87.7157358,12.5149977 L87.7157358,35.1638208 L81.0152282,35.1638208 L81.0152282,12.5149977 Z M11.2180264,12.5716929 C17.0453911,12.5716929 21.8552379,17.0154371 22.4273188,22.6910326 L15.4400709,22.6910326 C14.9217769,20.784737 13.1903773,19.4616421 11.2140961,19.4616421 C9.23781493,19.4616421 7.50641527,20.784737 6.98812126,22.6910326 L6.91650197,22.981753 L6.91650197,35.3299491 L0.0165947137,35.3299491 L0.0165947137,22.6910326 L0,22.6910326 C0.00436702991,22.6412697 0.0113542778,22.5923797 0.0165947137,22.5434898 L0.0165947137,0 L6.91650197,0 L6.91650197,13.4246473 C8.28018268,12.8600325 9.74198978,12.570169 11.2180264,12.5716929 Z M108.306903,19.4569596 L108.315643,19.4569596 C105.910484,19.4908081 103.978508,21.4469456 103.978508,23.8483362 C103.978508,26.2497267 105.910484,28.2058643 108.315643,28.2397127 L108.315643,35.125426 C105.06582,35.1264609 101.973734,33.7269236 99.8328939,31.2859711 C97.6907196,33.7498118 94.5831631,35.1645857 91.315182,35.1638208 L91.315182,28.2772346 C93.7442786,28.2772346 95.7134505,26.3113437 95.7134505,23.8862944 C95.7134505,21.4612451 93.7442786,19.4953542 91.315182,19.4953542 L91.315182,12.6087683 C94.5633322,12.6096753 97.6533072,14.008978 99.7935612,16.4482232 C101.934735,13.9855327 105.040447,12.5708729 108.306903,12.5703737 L108.306903,19.4569596 Z M115.5976,35.6068296 C113.166222,35.6068296 111.1952,33.6358082 111.1952,31.20443 C111.1952,28.7730518 113.166222,26.8020304 115.5976,26.8020304 C118.028978,26.8020304 120,28.7730518 120,31.20443 C120,33.6358082 118.028978,35.6068296 115.5976,35.6068296 Z" fill="url(#radialGradient-2)"></path>
              <path d="M52.5400582,12.834102 C58.439645,12.834102 63.2395014,17.868213 63.2395014,24.0547741 C63.2395014,30.2413351 58.439645,35.274573 52.5400582,35.274573 C50.8469303,35.2746382 49.1802607,34.8546496 47.6895391,34.0522776 L47.6895391,35.1741702 L41.1444392,35.1741702 L41.1444392,0 L47.6895391,0 L47.6895391,14.0563974 C49.1802607,13.2540254 50.8469303,12.8340368 52.5400582,12.834102 Z M52.5400582,28.4122573 C54.8329832,28.4122573 56.697022,26.4565846 56.697022,24.0530279 C56.697022,21.6494713 54.8321097,19.6937986 52.5400582,19.6937986 C50.2480066,19.6937986 48.3830943,21.6485982 48.3830943,24.0530279 C48.3830943,26.4574576 50.2480066,28.4122573 52.5400582,28.4122573 Z M72.6482086,12.6811259 L72.6482086,13.8872144 C74.1374669,13.109131 75.790596,12.6960273 77.471158,12.681998 L77.471158,19.5653356 C74.8549807,19.6185326 72.7355018,21.8737348 72.6482086,24.6766041 L72.6482086,35.3299491 L66.0636823,35.3299491 L66.0636823,12.6811259 L72.6482086,12.6811259 Z M38.1235798,12.6257498 L38.1218341,24.9499075 L38.15413,24.9499075 C38.1445285,25.0493663 38.1340542,25.148825 38.1218341,25.2474113 L38.1218341,48.6202121 L31.2183678,48.6202121 L31.2183678,34.2946657 C29.8617641,34.8533559 28.4085877,35.1402952 26.9413435,35.1391924 C21.0905488,35.1391924 16.2679857,30.658315 15.7268112,24.9499075 L22.6974879,24.9499075 C23.1855033,26.8844709 24.9212773,28.2438906 26.9173121,28.2547803 C28.913347,28.2656701 30.6638644,26.9252704 31.1729789,24.9961471 L31.1197343,24.9961471 L31.1101329,24.9533973 L31.2148763,24.5285166 L31.2201135,24.7169647 L31.2201135,12.6257498 L38.1235798,12.6257498 Z M81.0152282,12.5149977 L87.7157358,12.5149977 L87.7157358,35.1638208 L81.0152282,35.1638208 L81.0152282,12.5149977 Z M11.2180264,12.5716929 C17.0453911,12.5716929 21.8552379,17.0154371 22.4273188,22.6910326 L15.4400709,22.6910326 C14.9217769,20.784737 13.1903773,19.4616421 11.2140961,19.4616421 C9.23781493,19.4616421 7.50641527,20.784737 6.98812126,22.6910326 L6.91650197,22.981753 L6.91650197,35.3299491 L0.0165947137,35.3299491 L0.0165947137,22.6910326 L0,22.6910326 C0.00436702991,22.6412697 0.0113542778,22.5923797 0.0165947137,22.5434898 L0.0165947137,0 L6.91650197,0 L6.91650197,13.4246473 C8.28018268,12.8600325 9.74198978,12.570169 11.2180264,12.5716929 Z M108.306903,19.4569596 L108.315643,19.4569596 C105.910484,19.4908081 103.978508,21.4469456 103.978508,23.8483362 C103.978508,26.2497267 105.910484,28.2058643 108.315643,28.2397127 L108.315643,35.125426 C105.06582,35.1264609 101.973734,33.7269236 99.8328939,31.2859711 C97.6907196,33.7498118 94.5831631,35.1645857 91.315182,35.1638208 L91.315182,28.2772346 C93.7442786,28.2772346 95.7134505,26.3113437 95.7134505,23.8862944 C95.7134505,21.4612451 93.7442786,19.4953542 91.315182,19.4953542 L91.315182,12.6087683 C94.5633322,12.6096753 97.6533072,14.008978 99.7935612,16.4482232 C101.934735,13.9855327 105.040447,12.5708729 108.306903,12.5703737 L108.306903,19.4569596 Z M115.5976,35.6068296 C113.166222,35.6068296 111.1952,33.6358082 111.1952,31.20443 C111.1952,28.7730518 113.166222,26.8020304 115.5976,26.8020304 C118.028978,26.8020304 120,28.7730518 120,31.20443 C120,33.6358082 118.028978,35.6068296 115.5976,35.6068296 Z" fill="url(#radialGradient-3)"></path>
              <path d="M52.5400582,12.834102 C58.439645,12.834102 63.2395014,17.868213 63.2395014,24.0547741 C63.2395014,30.2413351 58.439645,35.274573 52.5400582,35.274573 C50.8469303,35.2746382 49.1802607,34.8546496 47.6895391,34.0522776 L47.6895391,35.1741702 L41.1444392,35.1741702 L41.1444392,0 L47.6895391,0 L47.6895391,14.0563974 C49.1802607,13.2540254 50.8469303,12.8340368 52.5400582,12.834102 Z M52.5400582,28.4122573 C54.8329832,28.4122573 56.697022,26.4565846 56.697022,24.0530279 C56.697022,21.6494713 54.8321097,19.6937986 52.5400582,19.6937986 C50.2480066,19.6937986 48.3830943,21.6485982 48.3830943,24.0530279 C48.3830943,26.4574576 50.2480066,28.4122573 52.5400582,28.4122573 Z M72.6482086,12.6811259 L72.6482086,13.8872144 C74.1374669,13.109131 75.790596,12.6960273 77.471158,12.681998 L77.471158,19.5653356 C74.8549807,19.6185326 72.7355018,21.8737348 72.6482086,24.6766041 L72.6482086,35.3299491 L66.0636823,35.3299491 L66.0636823,12.6811259 L72.6482086,12.6811259 Z M38.1235798,12.6257498 L38.1218341,24.9499075 L38.15413,24.9499075 C38.1445285,25.0493663 38.1340542,25.148825 38.1218341,25.2474113 L38.1218341,48.6202121 L31.2183678,48.6202121 L31.2183678,34.2946657 C29.8617641,34.8533559 28.4085877,35.1402952 26.9413435,35.1391924 C21.0905488,35.1391924 16.2679857,30.658315 15.7268112,24.9499075 L22.6974879,24.9499075 C23.1855033,26.8844709 24.9212773,28.2438906 26.9173121,28.2547803 C28.913347,28.2656701 30.6638644,26.9252704 31.1729789,24.9961471 L31.1197343,24.9961471 L31.1101329,24.9533973 L31.2148763,24.5285166 L31.2201135,24.7169647 L31.2201135,12.6257498 L38.1235798,12.6257498 Z M81.0152282,12.5149977 L87.7157358,12.5149977 L87.7157358,35.1638208 L81.0152282,35.1638208 L81.0152282,12.5149977 Z M11.2180264,12.5716929 C17.0453911,12.5716929 21.8552379,17.0154371 22.4273188,22.6910326 L15.4400709,22.6910326 C14.9217769,20.784737 13.1903773,19.4616421 11.2140961,19.4616421 C9.23781493,19.4616421 7.50641527,20.784737 6.98812126,22.6910326 L6.91650197,22.981753 L6.91650197,35.3299491 L0.0165947137,35.3299491 L0.0165947137,22.6910326 L0,22.6910326 C0.00436702991,22.6412697 0.0113542778,22.5923797 0.0165947137,22.5434898 L0.0165947137,0 L6.91650197,0 L6.91650197,13.4246473 C8.28018268,12.8600325 9.74198978,12.570169 11.2180264,12.5716929 Z M108.306903,19.4569596 L108.315643,19.4569596 C105.910484,19.4908081 103.978508,21.4469456 103.978508,23.8483362 C103.978508,26.2497267 105.910484,28.2058643 108.315643,28.2397127 L108.315643,35.125426 C105.06582,35.1264609 101.973734,33.7269236 99.8328939,31.2859711 C97.6907196,33.7498118 94.5831631,35.1645857 91.315182,35.1638208 L91.315182,28.2772346 C93.7442786,28.2772346 95.7134505,26.3113437 95.7134505,23.8862944 C95.7134505,21.4612451 93.7442786,19.4953542 91.315182,19.4953542 L91.315182,12.6087683 C94.5633322,12.6096753 97.6533072,14.008978 99.7935612,16.4482232 C101.934735,13.9855327 105.040447,12.5708729 108.306903,12.5703737 L108.306903,19.4569596 Z M115.5976,35.6068296 C113.166222,35.6068296 111.1952,33.6358082 111.1952,31.20443 C111.1952,28.7730518 113.166222,26.8020304 115.5976,26.8020304 C118.028978,26.8020304 120,28.7730518 120,31.20443 C120,33.6358082 118.028978,35.6068296 115.5976,35.6068296 Z" fill="url(#radialGradient-4)"></path>
              <path d="M52.5400582,12.834102 C58.439645,12.834102 63.2395014,17.868213 63.2395014,24.0547741 C63.2395014,30.2413351 58.439645,35.274573 52.5400582,35.274573 C50.8469303,35.2746382 49.1802607,34.8546496 47.6895391,34.0522776 L47.6895391,35.1741702 L41.1444392,35.1741702 L41.1444392,0 L47.6895391,0 L47.6895391,14.0563974 C49.1802607,13.2540254 50.8469303,12.8340368 52.5400582,12.834102 Z M52.5400582,28.4122573 C54.8329832,28.4122573 56.697022,26.4565846 56.697022,24.0530279 C56.697022,21.6494713 54.8321097,19.6937986 52.5400582,19.6937986 C50.2480066,19.6937986 48.3830943,21.6485982 48.3830943,24.0530279 C48.3830943,26.4574576 50.2480066,28.4122573 52.5400582,28.4122573 Z M72.6482086,12.6811259 L72.6482086,13.8872144 C74.1374669,13.109131 75.790596,12.6960273 77.471158,12.681998 L77.471158,19.5653356 C74.8549807,19.6185326 72.7355018,21.8737348 72.6482086,24.6766041 L72.6482086,35.3299491 L66.0636823,35.3299491 L66.0636823,12.6811259 L72.6482086,12.6811259 Z M38.1235798,12.6257498 L38.1218341,24.9499075 L38.15413,24.9499075 C38.1445285,25.0493663 38.1340542,25.148825 38.1218341,25.2474113 L38.1218341,48.6202121 L31.2183678,48.6202121 L31.2183678,34.2946657 C29.8617641,34.8533559 28.4085877,35.1402952 26.9413435,35.1391924 C21.0905488,35.1391924 16.2679857,30.658315 15.7268112,24.9499075 L22.6974879,24.9499075 C23.1855033,26.8844709 24.9212773,28.2438906 26.9173121,28.2547803 C28.913347,28.2656701 30.6638644,26.9252704 31.1729789,24.9961471 L31.1197343,24.9961471 L31.1101329,24.9533973 L31.2148763,24.5285166 L31.2201135,24.7169647 L31.2201135,12.6257498 L38.1235798,12.6257498 Z M81.0152282,12.5149977 L87.7157358,12.5149977 L87.7157358,35.1638208 L81.0152282,35.1638208 L81.0152282,12.5149977 Z M11.2180264,12.5716929 C17.0453911,12.5716929 21.8552379,17.0154371 22.4273188,22.6910326 L15.4400709,22.6910326 C14.9217769,20.784737 13.1903773,19.4616421 11.2140961,19.4616421 C9.23781493,19.4616421 7.50641527,20.784737 6.98812126,22.6910326 L6.91650197,22.981753 L6.91650197,35.3299491 L0.0165947137,35.3299491 L0.0165947137,22.6910326 L0,22.6910326 C0.00436702991,22.6412697 0.0113542778,22.5923797 0.0165947137,22.5434898 L0.0165947137,0 L6.91650197,0 L6.91650197,13.4246473 C8.28018268,12.8600325 9.74198978,12.570169 11.2180264,12.5716929 Z M108.306903,19.4569596 L108.315643,19.4569596 C105.910484,19.4908081 103.978508,21.4469456 103.978508,23.8483362 C103.978508,26.2497267 105.910484,28.2058643 108.315643,28.2397127 L108.315643,35.125426 C105.06582,35.1264609 101.973734,33.7269236 99.8328939,31.2859711 C97.6907196,33.7498118 94.5831631,35.1645857 91.315182,35.1638208 L91.315182,28.2772346 C93.7442786,28.2772346 95.7134505,26.3113437 95.7134505,23.8862944 C95.7134505,21.4612451 93.7442786,19.4953542 91.315182,19.4953542 L91.315182,12.6087683 C94.5633322,12.6096753 97.6533072,14.008978 99.7935612,16.4482232 C101.934735,13.9855327 105.040447,12.5708729 108.306903,12.5703737 L108.306903,19.4569596 Z M115.5976,35.6068296 C113.166222,35.6068296 111.1952,33.6358082 111.1952,31.20443 C111.1952,28.7730518 113.166222,26.8020304 115.5976,26.8020304 C118.028978,26.8020304 120,28.7730518 120,31.20443 C120,33.6358082 118.028978,35.6068296 115.5976,35.6068296 Z" fill="url(#radialGradient-5)"></path>
              <path d="M52.5400582,12.834102 C58.439645,12.834102 63.2395014,17.868213 63.2395014,24.0547741 C63.2395014,30.2413351 58.439645,35.274573 52.5400582,35.274573 C50.8469303,35.2746382 49.1802607,34.8546496 47.6895391,34.0522776 L47.6895391,35.1741702 L41.1444392,35.1741702 L41.1444392,0 L47.6895391,0 L47.6895391,14.0563974 C49.1802607,13.2540254 50.8469303,12.8340368 52.5400582,12.834102 Z M52.5400582,28.4122573 C54.8329832,28.4122573 56.697022,26.4565846 56.697022,24.0530279 C56.697022,21.6494713 54.8321097,19.6937986 52.5400582,19.6937986 C50.2480066,19.6937986 48.3830943,21.6485982 48.3830943,24.0530279 C48.3830943,26.4574576 50.2480066,28.4122573 52.5400582,28.4122573 Z M72.6482086,12.6811259 L72.6482086,13.8872144 C74.1374669,13.109131 75.790596,12.6960273 77.471158,12.681998 L77.471158,19.5653356 C74.8549807,19.6185326 72.7355018,21.8737348 72.6482086,24.6766041 L72.6482086,35.3299491 L66.0636823,35.3299491 L66.0636823,12.6811259 L72.6482086,12.6811259 Z M38.1235798,12.6257498 L38.1218341,24.9499075 L38.15413,24.9499075 C38.1445285,25.0493663 38.1340542,25.148825 38.1218341,25.2474113 L38.1218341,48.6202121 L31.2183678,48.6202121 L31.2183678,34.2946657 C29.8617641,34.8533559 28.4085877,35.1402952 26.9413435,35.1391924 C21.0905488,35.1391924 16.2679857,30.658315 15.7268112,24.9499075 L22.6974879,24.9499075 C23.1855033,26.8844709 24.9212773,28.2438906 26.9173121,28.2547803 C28.913347,28.2656701 30.6638644,26.9252704 31.1729789,24.9961471 L31.1197343,24.9961471 L31.1101329,24.9533973 L31.2148763,24.5285166 L31.2201135,24.7169647 L31.2201135,12.6257498 L38.1235798,12.6257498 Z M81.0152282,12.5149977 L87.7157358,12.5149977 L87.7157358,35.1638208 L81.0152282,35.1638208 L81.0152282,12.5149977 Z M11.2180264,12.5716929 C17.0453911,12.5716929 21.8552379,17.0154371 22.4273188,22.6910326 L15.4400709,22.6910326 C14.9217769,20.784737 13.1903773,19.4616421 11.2140961,19.4616421 C9.23781493,19.4616421 7.50641527,20.784737 6.98812126,22.6910326 L6.91650197,22.981753 L6.91650197,35.3299491 L0.0165947137,35.3299491 L0.0165947137,22.6910326 L0,22.6910326 C0.00436702991,22.6412697 0.0113542778,22.5923797 0.0165947137,22.5434898 L0.0165947137,0 L6.91650197,0 L6.91650197,13.4246473 C8.28018268,12.8600325 9.74198978,12.570169 11.2180264,12.5716929 Z M108.306903,19.4569596 L108.315643,19.4569596 C105.910484,19.4908081 103.978508,21.4469456 103.978508,23.8483362 C103.978508,26.2497267 105.910484,28.2058643 108.315643,28.2397127 L108.315643,35.125426 C105.06582,35.1264609 101.973734,33.7269236 99.8328939,31.2859711 C97.6907196,33.7498118 94.5831631,35.1645857 91.315182,35.1638208 L91.315182,28.2772346 C93.7442786,28.2772346 95.7134505,26.3113437 95.7134505,23.8862944 C95.7134505,21.4612451 93.7442786,19.4953542 91.315182,19.4953542 L91.315182,12.6087683 C94.5633322,12.6096753 97.6533072,14.008978 99.7935612,16.4482232 C101.934735,13.9855327 105.040447,12.5708729 108.306903,12.5703737 L108.306903,19.4569596 Z M115.5976,35.6068296 C113.166222,35.6068296 111.1952,33.6358082 111.1952,31.20443 C111.1952,28.7730518 113.166222,26.8020304 115.5976,26.8020304 C118.028978,26.8020304 120,28.7730518 120,31.20443 C120,33.6358082 118.028978,35.6068296 115.5976,35.6068296 Z" fill="url(#radialGradient-6)"></path>
              <path d="M52.5400582,12.834102 C58.439645,12.834102 63.2395014,17.868213 63.2395014,24.0547741 C63.2395014,30.2413351 58.439645,35.274573 52.5400582,35.274573 C50.8469303,35.2746382 49.1802607,34.8546496 47.6895391,34.0522776 L47.6895391,35.1741702 L41.1444392,35.1741702 L41.1444392,0 L47.6895391,0 L47.6895391,14.0563974 C49.1802607,13.2540254 50.8469303,12.8340368 52.5400582,12.834102 Z M52.5400582,28.4122573 C54.8329832,28.4122573 56.697022,26.4565846 56.697022,24.0530279 C56.697022,21.6494713 54.8321097,19.6937986 52.5400582,19.6937986 C50.2480066,19.6937986 48.3830943,21.6485982 48.3830943,24.0530279 C48.3830943,26.4574576 50.2480066,28.4122573 52.5400582,28.4122573 Z M72.6482086,12.6811259 L72.6482086,13.8872144 C74.1374669,13.109131 75.790596,12.6960273 77.471158,12.681998 L77.471158,19.5653356 C74.8549807,19.6185326 72.7355018,21.8737348 72.6482086,24.6766041 L72.6482086,35.3299491 L66.0636823,35.3299491 L66.0636823,12.6811259 L72.6482086,12.6811259 Z M38.1235798,12.6257498 L38.1218341,24.9499075 L38.15413,24.9499075 C38.1445285,25.0493663 38.1340542,25.148825 38.1218341,25.2474113 L38.1218341,48.6202121 L31.2183678,48.6202121 L31.2183678,34.2946657 C29.8617641,34.8533559 28.4085877,35.1402952 26.9413435,35.1391924 C21.0905488,35.1391924 16.2679857,30.658315 15.7268112,24.9499075 L22.6974879,24.9499075 C23.1855033,26.8844709 24.9212773,28.2438906 26.9173121,28.2547803 C28.913347,28.2656701 30.6638644,26.9252704 31.1729789,24.9961471 L31.1197343,24.9961471 L31.1101329,24.9533973 L31.2148763,24.5285166 L31.2201135,24.7169647 L31.2201135,12.6257498 L38.1235798,12.6257498 Z M81.0152282,12.5149977 L87.7157358,12.5149977 L87.7157358,35.1638208 L81.0152282,35.1638208 L81.0152282,12.5149977 Z M11.2180264,12.5716929 C17.0453911,12.5716929 21.8552379,17.0154371 22.4273188,22.6910326 L15.4400709,22.6910326 C14.9217769,20.784737 13.1903773,19.4616421 11.2140961,19.4616421 C9.23781493,19.4616421 7.50641527,20.784737 6.98812126,22.6910326 L6.91650197,22.981753 L6.91650197,35.3299491 L0.0165947137,35.3299491 L0.0165947137,22.6910326 L0,22.6910326 C0.00436702991,22.6412697 0.0113542778,22.5923797 0.0165947137,22.5434898 L0.0165947137,0 L6.91650197,0 L6.91650197,13.4246473 C8.28018268,12.8600325 9.74198978,12.570169 11.2180264,12.5716929 Z M108.306903,19.4569596 L108.315643,19.4569596 C105.910484,19.4908081 103.978508,21.4469456 103.978508,23.8483362 C103.978508,26.2497267 105.910484,28.2058643 108.315643,28.2397127 L108.315643,35.125426 C105.06582,35.1264609 101.973734,33.7269236 99.8328939,31.2859711 C97.6907196,33.7498118 94.5831631,35.1645857 91.315182,35.1638208 L91.315182,28.2772346 C93.7442786,28.2772346 95.7134505,26.3113437 95.7134505,23.8862944 C95.7134505,21.4612451 93.7442786,19.4953542 91.315182,19.4953542 L91.315182,12.6087683 C94.5633322,12.6096753 97.6533072,14.008978 99.7935612,16.4482232 C101.934735,13.9855327 105.040447,12.5708729 108.306903,12.5703737 L108.306903,19.4569596 Z M115.5976,35.6068296 C113.166222,35.6068296 111.1952,33.6358082 111.1952,31.20443 C111.1952,28.7730518 113.166222,26.8020304 115.5976,26.8020304 C118.028978,26.8020304 120,28.7730518 120,31.20443 C120,33.6358082 118.028978,35.6068296 115.5976,35.6068296 Z" fill="url(#radialGradient-7)"></path>
              <path d="M52.5400582,12.834102 C58.439645,12.834102 63.2395014,17.868213 63.2395014,24.0547741 C63.2395014,30.2413351 58.439645,35.274573 52.5400582,35.274573 C50.8469303,35.2746382 49.1802607,34.8546496 47.6895391,34.0522776 L47.6895391,35.1741702 L41.1444392,35.1741702 L41.1444392,0 L47.6895391,0 L47.6895391,14.0563974 C49.1802607,13.2540254 50.8469303,12.8340368 52.5400582,12.834102 Z M52.5400582,28.4122573 C54.8329832,28.4122573 56.697022,26.4565846 56.697022,24.0530279 C56.697022,21.6494713 54.8321097,19.6937986 52.5400582,19.6937986 C50.2480066,19.6937986 48.3830943,21.6485982 48.3830943,24.0530279 C48.3830943,26.4574576 50.2480066,28.4122573 52.5400582,28.4122573 Z M72.6482086,12.6811259 L72.6482086,13.8872144 C74.1374669,13.109131 75.790596,12.6960273 77.471158,12.681998 L77.471158,19.5653356 C74.8549807,19.6185326 72.7355018,21.8737348 72.6482086,24.6766041 L72.6482086,35.3299491 L66.0636823,35.3299491 L66.0636823,12.6811259 L72.6482086,12.6811259 Z M38.1235798,12.6257498 L38.1218341,24.9499075 L38.15413,24.9499075 C38.1445285,25.0493663 38.1340542,25.148825 38.1218341,25.2474113 L38.1218341,48.6202121 L31.2183678,48.6202121 L31.2183678,34.2946657 C29.8617641,34.8533559 28.4085877,35.1402952 26.9413435,35.1391924 C21.0905488,35.1391924 16.2679857,30.658315 15.7268112,24.9499075 L22.6974879,24.9499075 C23.1855033,26.8844709 24.9212773,28.2438906 26.9173121,28.2547803 C28.913347,28.2656701 30.6638644,26.9252704 31.1729789,24.9961471 L31.1197343,24.9961471 L31.1101329,24.9533973 L31.2148763,24.5285166 L31.2201135,24.7169647 L31.2201135,12.6257498 L38.1235798,12.6257498 Z M81.0152282,12.5149977 L87.7157358,12.5149977 L87.7157358,35.1638208 L81.0152282,35.1638208 L81.0152282,12.5149977 Z M11.2180264,12.5716929 C17.0453911,12.5716929 21.8552379,17.0154371 22.4273188,22.6910326 L15.4400709,22.6910326 C14.9217769,20.784737 13.1903773,19.4616421 11.2140961,19.4616421 C9.23781493,19.4616421 7.50641527,20.784737 6.98812126,22.6910326 L6.91650197,22.981753 L6.91650197,35.3299491 L0.0165947137,35.3299491 L0.0165947137,22.6910326 L0,22.6910326 C0.00436702991,22.6412697 0.0113542778,22.5923797 0.0165947137,22.5434898 L0.0165947137,0 L6.91650197,0 L6.91650197,13.4246473 C8.28018268,12.8600325 9.74198978,12.570169 11.2180264,12.5716929 Z M108.306903,19.4569596 L108.315643,19.4569596 C105.910484,19.4908081 103.978508,21.4469456 103.978508,23.8483362 C103.978508,26.2497267 105.910484,28.2058643 108.315643,28.2397127 L108.315643,35.125426 C105.06582,35.1264609 101.973734,33.7269236 99.8328939,31.2859711 C97.6907196,33.7498118 94.5831631,35.1645857 91.315182,35.1638208 L91.315182,28.2772346 C93.7442786,28.2772346 95.7134505,26.3113437 95.7134505,23.8862944 C95.7134505,21.4612451 93.7442786,19.4953542 91.315182,19.4953542 L91.315182,12.6087683 C94.5633322,12.6096753 97.6533072,14.008978 99.7935612,16.4482232 C101.934735,13.9855327 105.040447,12.5708729 108.306903,12.5703737 L108.306903,19.4569596 Z M115.5976,35.6068296 C113.166222,35.6068296 111.1952,33.6358082 111.1952,31.20443 C111.1952,28.7730518 113.166222,26.8020304 115.5976,26.8020304 C118.028978,26.8020304 120,28.7730518 120,31.20443 C120,33.6358082 118.028978,35.6068296 115.5976,35.6068296 Z" fill="url(#radialGradient-8)"></path>
              <path d="M52.5400582,12.834102 C58.439645,12.834102 63.2395014,17.868213 63.2395014,24.0547741 C63.2395014,30.2413351 58.439645,35.274573 52.5400582,35.274573 C50.8469303,35.2746382 49.1802607,34.8546496 47.6895391,34.0522776 L47.6895391,35.1741702 L41.1444392,35.1741702 L41.1444392,0 L47.6895391,0 L47.6895391,14.0563974 C49.1802607,13.2540254 50.8469303,12.8340368 52.5400582,12.834102 Z M52.5400582,28.4122573 C54.8329832,28.4122573 56.697022,26.4565846 56.697022,24.0530279 C56.697022,21.6494713 54.8321097,19.6937986 52.5400582,19.6937986 C50.2480066,19.6937986 48.3830943,21.6485982 48.3830943,24.0530279 C48.3830943,26.4574576 50.2480066,28.4122573 52.5400582,28.4122573 Z M72.6482086,12.6811259 L72.6482086,13.8872144 C74.1374669,13.109131 75.790596,12.6960273 77.471158,12.681998 L77.471158,19.5653356 C74.8549807,19.6185326 72.7355018,21.8737348 72.6482086,24.6766041 L72.6482086,35.3299491 L66.0636823,35.3299491 L66.0636823,12.6811259 L72.6482086,12.6811259 Z M38.1235798,12.6257498 L38.1218341,24.9499075 L38.15413,24.9499075 C38.1445285,25.0493663 38.1340542,25.148825 38.1218341,25.2474113 L38.1218341,48.6202121 L31.2183678,48.6202121 L31.2183678,34.2946657 C29.8617641,34.8533559 28.4085877,35.1402952 26.9413435,35.1391924 C21.0905488,35.1391924 16.2679857,30.658315 15.7268112,24.9499075 L22.6974879,24.9499075 C23.1855033,26.8844709 24.9212773,28.2438906 26.9173121,28.2547803 C28.913347,28.2656701 30.6638644,26.9252704 31.1729789,24.9961471 L31.1197343,24.9961471 L31.1101329,24.9533973 L31.2148763,24.5285166 L31.2201135,24.7169647 L31.2201135,12.6257498 L38.1235798,12.6257498 Z M81.0152282,12.5149977 L87.7157358,12.5149977 L87.7157358,35.1638208 L81.0152282,35.1638208 L81.0152282,12.5149977 Z M11.2180264,12.5716929 C17.0453911,12.5716929 21.8552379,17.0154371 22.4273188,22.6910326 L15.4400709,22.6910326 C14.9217769,20.784737 13.1903773,19.4616421 11.2140961,19.4616421 C9.23781493,19.4616421 7.50641527,20.784737 6.98812126,22.6910326 L6.91650197,22.981753 L6.91650197,35.3299491 L0.0165947137,35.3299491 L0.0165947137,22.6910326 L0,22.6910326 C0.00436702991,22.6412697 0.0113542778,22.5923797 0.0165947137,22.5434898 L0.0165947137,0 L6.91650197,0 L6.91650197,13.4246473 C8.28018268,12.8600325 9.74198978,12.570169 11.2180264,12.5716929 Z M108.306903,19.4569596 L108.315643,19.4569596 C105.910484,19.4908081 103.978508,21.4469456 103.978508,23.8483362 C103.978508,26.2497267 105.910484,28.2058643 108.315643,28.2397127 L108.315643,35.125426 C105.06582,35.1264609 101.973734,33.7269236 99.8328939,31.2859711 C97.6907196,33.7498118 94.5831631,35.1645857 91.315182,35.1638208 L91.315182,28.2772346 C93.7442786,28.2772346 95.7134505,26.3113437 95.7134505,23.8862944 C95.7134505,21.4612451 93.7442786,19.4953542 91.315182,19.4953542 L91.315182,12.6087683 C94.5633322,12.6096753 97.6533072,14.008978 99.7935612,16.4482232 C101.934735,13.9855327 105.040447,12.5708729 108.306903,12.5703737 L108.306903,19.4569596 Z M115.5976,35.6068296 C113.166222,35.6068296 111.1952,33.6358082 111.1952,31.20443 C111.1952,28.7730518 113.166222,26.8020304 115.5976,26.8020304 C118.028978,26.8020304 120,28.7730518 120,31.20443 C120,33.6358082 118.028978,35.6068296 115.5976,35.6068296 Z" fill="url(#radialGradient-9)"></path>
              <circle fill="#68B6C7" cx="115.5976" cy="31.20443" r="4.40239962"></circle>
            </g>
          </g>
      </svg>
  </div>
  <nav class="navModuleNavigation">
    ${mkLinks()}
  </nav>
  </header>`;

  const div = document.createElement('DIV');
  div.classList.add('heading');
  document.body.insertAdjacentElement('afterbegin', div);
  document.querySelector('.heading').innerHTML = header;
}

function mkLinks () {
  /*
  const links = [
    { path: '', name: 'documentation' },
    { path: '/s/web-wallet/', name: 'wallet' },
    { path: '/s/web-blockexplorer/', name: 'explorer' }
  ];
  return links.reduce((htmlStr, link) => `${htmlStr}<a href="${determineHost() + link.path}">${link.name}</a>`, '');
  */
  return '<a href="#" id="loginButton">Connect to wallet</a>';
}

function checkCookie () {
  const cookie = getCookie();
  if (cookie && cookie['private-token']) privateToken = cookie['private-token'];
}

window.onload = () => {
  checkCookie();
  renderHeader();

  /* DEPRECATED: NO LONGER RENDER ALL ENDPOINTS
  fetchQuery('/meta')
    .then(meta => menuWithEndpoints(meta, '_ui'))
    .then(renderMenu)
    .catch(e => console.log('error:', e));
  */

};
