//
// hybrixd - apiqueue.js
// API/Transaction queue processor
//

const sequential = require('../util/sequential');
const hosts = require('./hosts');
const fs = require('fs');
const xml2js = require('xml2js');
const DEFAULT_APIREQUEST_TIMEOUT = 24000; // Note: should  be less than process timeout
const DEFAULT_APICACHE_TIME = 8000;
const DEFAULT_RETRY_LIMIT = 3;
const DEFAULT_THROTTLE = 10;
const TIMER_DEFAULT = 10;

// API queue
const APIqueue = {};
let APIqueueInterval = null; // the API queue main interval loop
let APIqueueIntervalBusy = false; // blocks execution of next interval while previous one is running
let APIqueueInitiated = false; // whether the API queue has been initiated

const testDataFile = './APIqueue/testData.json';
let testMode = false; // when in test mode the testData is used as fallback
let testModeForce = false; // when in test mode force the testData is used without checking for available data
const testData = {};

function escapeJSON (str) {
  return str
    .replace(/[\\]/g, '\\\\')
    .replace(/["]/g, '\\"')
    .replace(/[/]/g, '\\/')
    .replace(/[\b]/g, '\\b')
    .replace(/[\f]/g, '\\f')
    .replace(/[\n]/g, '\\n')
    .replace(/[\r]/g, '\\r')
    .replace(/[\t]/g, '\\t');
}

function truncateString(str, num) {
  if (str.length > num) return `${str.slice(0, num)}…`;
  else return str;
}

function getTestQuery (APIrequest) {
  const methodPOST = APIrequest.method === 'POST' || APIrequest.method === 'PUT' || APIrequest.method === 'PATCH';
  const tcpHost = (APIrequest.host + '').startsWith('tcp://') || (APIrequest.host + '').startsWith('tcps://');
  const wsHost = (APIrequest.host + '').startsWith('ws://') || (APIrequest.host + '').startsWith('wss://');
  if (methodPOST || tcpHost || wsHost) {
    const data = APIrequest.args.data;
    let generalizedData;
    if (typeof data === 'object' && data !== null) {
      generalizedData = {};
      for (const key in data) {
        if (key !== 'id') { // we need to remove identifiers
          generalizedData[key] = data[key];
        }
      }
    } else {
      generalizedData = data;
    }
    return escapeJSON(APIrequest.host + APIrequest.args.path + JSON.stringify(generalizedData));
  } else {
    return escapeJSON(APIrequest.host + APIrequest.args.path);
  }
}

function isTesting () { return testMode; }

function testStart (force) {
  testModeForce = !!force;
  testMode = true;
  if (fs.existsSync(testDataFile)) {
    const newTestData = JSON.parse(fs.readFileSync(testDataFile, 'utf-8'));
    for (const query in newTestData) {
      testData[query] = newTestData[query];
    }
  }
}

function testStop () {
  testMode = false;
  testModeForce = false;
}

function testWrite (APIrequest, data) {
  const query = getTestQuery(APIrequest);
  global.hybrixd.logger(['test', 'apiQueue'], 'Write to test data: ', query);
  testData[query] = data;
  fs.writeFileSync(testDataFile, JSON.stringify(testData));
}

function testRead (APIrequest, qid) {
  const query = getTestQuery(APIrequest);
  if (testData.hasOwnProperty(query)) {
    global.hybrixd.logger(['test', 'apiQueue'], 'Read from test data: ', query);
    done(qid, testData[query], true);
    return true;
  } else {
    global.hybrixd.logger(['test', 'apiQueue'], 'No test data available for: ', query);
    return false;
  }
}

function isRunning () {
  return !!APIqueueInterval;
}

function isReady () {
  return APIqueueInitiated;
}

// pause the APIqueue
const pause = async (cbArr) => {
  cbArr = cbArr || [];
  if (isRunning()) global.hybrixd.logger(['info', 'apiQueue'], 'Paused queue');
  cbArr.unshift(hosts.closeAll);
  clearInterval(APIqueueInterval);
  APIqueueInterval = null;
  APIqueueInitiated = false;
  sequential.next(cbArr);
}

// resume the APIqueue
const resume =  async (cbArr) => {
  if (!isRunning()) {
    global.hybrixd.logger(['info', 'apiQueue'], 'Started queue');
    initialize(cbArr);
  } else sequential.next(cbArr);
}

const call = async (qid, APIrequest, host, now) => {
  APIrequest.retries++; // increase number of tries
  APIrequest.busy = true;
  const truncateLength = 128;
  function failCall(failHost,e) {
    if (APIqueue.hasOwnProperty(qid)) {
      APIqueue[qid].busy = false;
      APIqueue[qid].errorMessage = e;
    }
    const errorDetails = e ? ` -> ${JSON.stringify(e)}` : '';
    global.hybrixd.logger(['error', 'host', 'apiQueue'], `Routing call error for ${failHost}${APIrequest.args.path}, with no more alternatives${errorDetails}`);
    const failHosts = typeof APIrequest.host === 'string' ? APIrequest.host : `[${APIrequest.host.join(',')}]`;
    const errorWord = typeof APIrequest.host === 'string' ? 'error' : 'errors';
    return fail(qid, `Routing call ${errorWord} for ${failHosts}${APIrequest.args.path} ${APIrequest.args.data?truncateString(JSON.stringify(APIrequest.args.data),truncateLength)+' ':''}${errorDetails}`);
  }
  const errorCallback = e => {
    if (APIrequest.retries >= APIrequest.retry) {
      failCall(host,e);
    } else {
      const altHost = hosts.findAvailableHost(APIrequest, now);
      if (altHost) {
        const errorDetails = e ? ` -> ${JSON.stringify(e)}` : '';
        global.hybrixd.logger(['warn', 'host', 'apiQueue'], `Routing call error for ${host}${APIrequest.args.path}, trying alternative host ${altHost} ...${errorDetails}`);
        return call(qid, APIrequest, altHost, now);
      } else return failCall(host,e);
    }
  };
  return hosts.call(host, APIrequest, now, handleResult(qid), errorCallback);
}

const update = async (qid, APIrequest, now) => {
  const timestamp = APIrequest.time;
  const timeout = timestamp + APIrequest.timeout;
  if (testMode && testModeForce) { // skip the call, only use cached data
    if (testRead(APIrequest, qid)) {
      // cached data found
    } else return fail(qid, 'Test mode force with no test data available');
  } else {
    const errorOutput = `${APIrequest.host}${(typeof APIrequest.args === 'object' && APIrequest.args !== null ? APIrequest.args.path : '')} in /proc/${APIrequest.qrtzProcessStep.getProcessID()}`;
    if (!APIrequest.host || ((APIrequest.host instanceof Array) && APIrequest.host.length === 0)) {
      global.hybrixd.logger(['error', 'host', 'apiQueue'], `Empty or no host defined for ${errorOutput}.`);
      return fail(qid, 'No host defined for API call');
    } else if (!APIrequest.busy && (APIrequest.timeout ===0 || timeout > now) && (APIrequest.retry === 0 || APIrequest.retries < APIrequest.retry)) { // there is time and retries left, so the request is still valid
      const host = hosts.findAvailableHost(APIrequest, now);
      if (host) return call(qid, APIrequest, host, now);
    } else if (!APIrequest.busy && APIrequest.retries >= APIrequest.retry) { // no more retries left : delete the queue object and fail the process
      const errorDetails = APIrequest.errorMessage ? ` -> ${APIrequest.errorMessage}` : '';
      global.hybrixd.logger(['warn', 'host', 'apiQueue'], `Maximum API call retries exceeded for ${errorOutput}${errorDetails}`);
      return fail(qid, `Maximum API call retries exceeded -> ${APIrequest.errorMessage}`);
    } else if (timeout <= now) { // no more time left : delete the queue object and fail the process
      if (APIrequest.timeout === 0) {
        return done(qid, null);
      } else {
        global.hybrixd.logger(['warn', 'host', 'apiQueue'], `Time out for ${errorOutput}  (${APIrequest.timeout} ms)`);
        return fail(qid, 'Request timed out (' + APIrequest.timeout + ' ms)');
      }
    }
  }
}

const initialize = async (callbackArray) => {
  if (!APIqueueInitiated) {
    global.hybrixd.logger(['info', 'apiQueue'], 'Initialized queue');
    APIqueueInitiated = true;
    APIqueueInterval = setInterval(updateApiQueue, TIMER_DEFAULT);
  }
  return sequential.next(callbackArray);
}

const updateApiQueue = async () => {
  if (!APIqueueIntervalBusy) {
    APIqueueIntervalBusy = true;
    await Object.keys(APIqueue)
      .forEach(updateQueueItem(Date.now()));
    APIqueueIntervalBusy = false;
  }
  return;
}

const updateQueueItem = now => qid => update(qid, APIqueue[qid], now);

const handleResult = qid => async (data) => {
  if (APIqueue.hasOwnProperty(qid)) {
    const APIrequest = APIqueue[qid];
    switch (APIrequest.parsing) {
      case 'none' : break;
      case 'json' :
        if (typeof data === 'string') {
          try {
            data = JSON.parse(data);
          } catch (result) {
            global.hybrixd.logger(['error', 'apiQueue'], `Parsing "${APIrequest.parsing}" failed: (/proc/${APIrequest.qrtzProcessStep.getProcessID()}) ${data}`);
            return fail(qid, `JSON Parsing failed for ${APIrequest.host}`);
          }
        }
        break;
      case 'xml' :
      {
        const parser = new xml2js.Parser({async: true, parseBooleans: true, parseNumbers: true, attrkey: 'attributes' });
        return parser.parseString(data, (error, result) => {
          if (error) fail(qid, `XML parsing failed for ${APIrequest.host}`, error);
          else done(qid, result);
        });
      }
      case 'auto' :
      default:
        if (typeof data === 'string') {
          try {
            data = JSON.parse(data);
          } catch (e) {
            // FIXME API queue: no JSON, skipping parsing...
            global.hybrixd.logger(['warn', 'apiQueue'], `No JSON result, skipping parsing data from ${APIrequest.host}`);
          }
        }
        break;
    }
    return done(qid, data);
  }
  // on timeout ${qid} can be deleted, warning about this is deprecated!
  // else global.hybrixd.logger(['warn', 'apiQueue'], `Request ${qid} not found in queue!`);
};

const done = async (qid, data, skipWrite) => {
  const APIrequest = APIqueue[qid];
  if (testMode && !skipWrite) testWrite(APIrequest, data);
  delete APIqueue[qid];
  if (typeof APIrequest !== 'undefined' && typeof APIrequest.dataCallback === 'function') APIrequest.dataCallback(data);
}

const fail = async (qid, data) => {
  const APIrequest = APIqueue[qid];
  if (testMode && !testModeForce && testRead(APIrequest, qid)) {
    // skip failure, got fallback reply from test data
  } else {
    if (typeof APIrequest !== 'undefined' && typeof APIrequest.errorCallback === 'function') APIrequest.errorCallback(data);
    delete APIqueue[qid];
  }
}

const insert = async (queueobject) => {
  const suffix = `0000${Math.random() * 9999}`.slice(-4);
  const now = Date.now();
  const qid = `${now}${suffix}`;
  
  const queueobjectInit = { // sane defaults
    busy: false, // (not modifiable)
    time: now, // (not modifiable)
    retries: 0, // number of retries done (not modifiable)

    cache: DEFAULT_APICACHE_TIME,
    retry: DEFAULT_RETRY_LIMIT, // max nr of retries allowed
    throttle: DEFAULT_THROTTLE,
    timeout: DEFAULT_APIREQUEST_TIMEOUT,

    method: 'POST',
    host: null,
    args: null, // data passed to call (inluding args.headers, args.path and args.data)
    identifier: 'id', // identifier key used for websocket responses

    parsing: 'auto', // whether to parse the result as 'auto', 'json' or 'none' for no parsing. TODO: xml, yaml,
    ping: false, // only check if a socket connection to the host can be made

    qrtzProcessStep: null,
    dataCallback: null,
    errorCallback: null,
    ignore404: false, // whether to fail on 404
    ignoreError: false // whether to fail on error status (<200 || >=300)
  };

  const notModifiables = ['busy', 'time', 'retries'];
  if (queueobject && typeof queueobject === 'object' && queueobject !== null) notModifiables.forEach(deleteStatusWhenQueued(queueobject)); // remove from queueobject as they should only be set by APIqueue

  APIqueue[qid] = Object.assign(queueobjectInit, queueobject);
  const APIrequest = APIqueue[qid];
  if (APIrequest.ignore404) APIrequest.args.ignore404 = true;
  if (APIrequest.ignoreError) APIrequest.args.ignoreError = true;
  if (APIrequest.qrtzProcessStep) {
    const processTimeOut = APIrequest.qrtzProcessStep.getTimeOut();
    // LATENT BUG: for some reason APIrequest.timeout can go below zero here, so we catch that in the following if statement
    //             note this only happens when recipes are imported/merged more than two layers deep
    if (APIrequest.timeout < 0 || processTimeOut <= APIrequest.timeout) APIrequest.timeout = processTimeOut - TIMER_DEFAULT; // ensure api request timeout is always smaller than process timeout
  }
}

function deleteStatusWhenQueued (queue) {
  return status => {
    if (queue.hasOwnProperty(status)) delete queue[status];
  };
}

exports.initialize = initialize;
exports.add = insert;
exports.pause = pause;
exports.resume = resume;
exports.isRunning = isRunning;
exports.isReady = isReady;
exports.testStart = testStart;
exports.testStop = testStop;
exports.isTesting = isTesting;
