const DEFAULT_PROCESS_TIMEOUT = 30000; // 30 second default proc timeout
const DEFAULT_PROCESS_TIMEMAX = 3600000; // 1 hour default proc max timeout
const PROCESS_PURGE_TIME = 60; // process purge time (sec): one minute
const PROCESS_MAX_DEPTH = 32; // maximum times process may spawn nested children

const {createHash} = require('node:crypto');
const QrtzFunction = require('./function').QrtzFunction;
const Vars = require('./vars');
const modules = require('../modules');
const conf = require('../conf/conf');
const sequential = require('../util/sequential');
const queuedProcesses = {};
const busyProcesses = {};
const finishedProcesses = {};
const {rout} = require('../router/router.js');

const mimeTypesByExtension = {
  jpg: 'image/jpg',
  png: 'image/png',
  css: 'text/css',
  ico: 'image/x-icon',
  js: 'text/javascript',
  json: 'application/json',
  svg: 'image/svg+xml',
  html: 'text/html',
  ttf: 'application/x-font-ttf',
  woff2: 'application/x-font-woff',
  eot: 'application/vnd.ms-fontobject',
  lzma: 'data'  
};
const extensionsWithCache = ['jpg','png', 'css', 'ico', 'svg', 'ttf', 'woff2', 'eot'];

/**
 * @param processID
 */
function getProcess (processID) {
  if (queuedProcesses.hasOwnProperty(processID)) {
    return queuedProcesses[processID];
  } else if (busyProcesses.hasOwnProperty(processID)) {
    return busyProcesses[processID];
  } else if (finishedProcesses.hasOwnProperty(processID)) {
    return finishedProcesses[processID];
  } else {
    return null;
  }
}

/**
 * @param processID
 */
function get404 (processID) {
  return {id: processID, error: 404, data: 'Process not found'};
}

const get403 = (processID) => {
  return {id: processID, error: 403, data: 'Forbidden'};
};

// to expose and pass as first param to qrtz methods
/**
 * @param qrtzStatement
 * @param qrtzProcess
 * @param step
 */
function QrtzProcessStep (qrtzStatement, qrtzProcess, step) {
  let data = null;
  let childData = null;
  let error = 0;
  let progress = null;
  let started = null;
  let stopped = null;

  let childProcessIDCounter = 0;
  const busyChildProcesses = {};
  const finishedChildProcesses = {};

  this.toString = () => qrtzStatement.toString();

  this.peek = variableName => {
    return Vars.peek(this, variableName);
  };

  this.poke = (variableName, vdata) => {
    return Vars.poke(this, variableName, vdata);
  };

  this.reset = async (data_) => {
    data = data_;
    childData = null;
    error = 0;
    progress = null;
    started = null;
    stopped = null;
    for (const childID in busyChildProcesses) delete busyChildProcesses[childID];
    for (const childID in finishedChildProcesses) delete finishedChildProcesses[childID];
  };

  this.childReportsStop = async (error_, data_, childID, childReferenceID, postProcessCallback) => {
    // the childReferenceID is used to format the resulted data the way the parent expects it
    // for example a qrtz 'each doSomething' called on {a:1,b:2} returns {a:somethingDoneWith(1),b:somethingDoneWith(2)}
    // a and b are the childReferenceID. Their respective childIDs would be 0 and 1
    if (!busyChildProcesses.hasOwnProperty(childID)) {
      global.hybrixd.logger(['error', 'process'], `Orphan process /p/${this.getProcessID()}.${childID} not found. `);
      return;
    }
    const childProcess = busyChildProcesses[childID];

    const onlyChild = childReferenceID === true;

    if (onlyChild) data = data_;
    else {
      if (childData === null) childData = isNaN(childReferenceID) ? {} : [];

      if (isNaN(childReferenceID) && (childData instanceof Array)) { // if an array gets a non numeric key, transform to object
        childData = Object.fromEntries(childData.map((value, key) => [key, value]));
      }
      if (data_ instanceof Error) data_ = this.getSessionID() === 1 ? data_.stack.toString() : data_.toString();
      childData[childReferenceID] = data_;
      data = childData;
    }

    delete busyChildProcesses[childID];
    finishedChildProcesses[childID] = childProcess;

    if (typeof postProcessCallback === 'function') postProcessCallback(error_, data_);
    else if (Object.keys(busyChildProcesses).length === 0) { // if last child process has stopped
      let failed, errorData, errorCode;
      if (onlyChild) {
        failed = error_ !== 0;
        errorData = data_;
        errorCode = error_;
      } else {
        for (const childID in finishedChildProcesses) { // if any child process has failed, fail the entire process
          const childErrorCode = finishedChildProcesses[childID].getError();
          if (childErrorCode !== 0) {
            failed = true;
            errorCode = childErrorCode;
            errorData = childData;
            break;
          }
        }
      }
      if (failed) this.fail(errorCode, errorData);
      else this.next(data);
    }
  };

  this.hook = async (jumpOrErrOrData, err) => qrtzProcess.hook(step, jumpOrErrOrData, err);

  this.fork = async (commandOrFunction, ydata, childReferenceID, options) => {
    const procID = this.getProcessID();
    if (procID.split('.').length > PROCESS_MAX_DEPTH) { // max-depth to avoid runaway processes
      this.fail(`fork: Process '${procID}' halted for exceeding maximum call depth of ${PROCESS_MAX_DEPTH}`);
      return 0;
    }
    let command;
    let qrtzFunction;
    if (commandOrFunction instanceof Array) {
      command = commandOrFunction;
    } else if (commandOrFunction instanceof QrtzFunction) {
      qrtzFunction = commandOrFunction;
      command = [];
    } else {
      this.fail('fork: Illegal fork command or function');
      return 0;
    }
    let recipe;
    let expose = false; // however, assets and sources are always exposed
    if (command.length > 0 && command[0].includes('::')) { // handle calls to specifically exposed endpoints of other recipes
      const [scopeId, methodName] = command[0].split('::');
      const currentId = this.getRecipe().id;
      if (methodName === 'cron' || methodName === 'init') {
        this.fail(`fork: Module '${scopeId}' cron or init methods may not be called by '${currentId}'`);
        return 0;
      }
      // Load other recipe
      if (global.hybrixd.asset.hasOwnProperty(scopeId)) {
        recipe = global.hybrixd.asset[scopeId];
        expose = true;
      } else if (global.hybrixd.source.hasOwnProperty(scopeId)) {
        recipe = global.hybrixd.source[scopeId];
        expose = true;
      } else if (global.hybrixd.engine.hasOwnProperty(scopeId)) recipe = global.hybrixd.engine[scopeId];
      else {
        this.fail(`fork: Module '${scopeId}' could not be found`);
        return 0;
      }
      // check if other recipe exposes the requested method (asset engines are always exposed)
      if (!expose) {
        if (typeof recipe === 'object' && recipe !== null && recipe.hasOwnProperty('expose') &&
        typeof recipe.expose === 'object' && recipe.expose !== null && recipe.expose.hasOwnProperty(currentId)) {
          if (recipe.expose[currentId] !== methodName && !(recipe.expose[currentId] instanceof Array) && !recipe.expose[currentId].includes(methodName)) {
            this.fail(`fork: Module '${scopeId}' does not expose method '${methodName}' to module '${currentId}'`);
            return 0;
          }
        } else {
          this.fail(`fork: Module '${scopeId}' does not expose method '${methodName}' to module '${currentId}'`);
          return 0;
        }
      }
      command[0] = methodName; // 'scope::methodName' -> 'methodName'
    } else recipe = this.getRecipe();

    let subProcess;
    const sessionID = this.getSessionID();

    if (typeof childReferenceID !== 'undefined' && childReferenceID !== false) { // childProcess
      const vars = options && options.shareScope ? this.getVars() : {};
      const postProcessCallback = typeof options === 'object' && options !== null ? options.callback : undefined;
      const childID = childProcessIDCounter++;
      subProcess = new QrtzProcess({
        data: ydata,
        recipe,
        command,
        timeout: this.getTimeOut(),
        vars,
        function: qrtzFunction,
        sessionID,
        parent: this,
        childID,
        childReferenceID,
        postProcessCallback
      });
      busyChildProcesses[childID] = subProcess;
    } else { // independent process
      subProcess = new QrtzProcess({data: ydata, recipe, command, function: qrtzFunction, sessionID});
      if (childReferenceID !== false) return this.next(subProcess.getProcessID()); // if it's an independent process run from Qrtz we don't have to wait for it
    }
    return subProcess.getProcessID();
  };

  this.adopt = async (processID, childReferenceID, postProcessCallback) => {
    const childID = childProcessIDCounter++;
    const subProcess = getProcess(processID);
    if (subProcess && subProcess.getSessionID() === this.getSessionID()) {
      subProcess.adopt({parent: this, childID, childReferenceID, postProcessCallback});
      busyChildProcesses[childID] = subProcess;
    } else {
      this.fail(`Could not adopt process ${processID}`);
    }
    return;
  };

  this.read = async (processID, childReferenceID, dataCallback, errorCallback) => {
    let procID = processID;
    const callback = (error, data) => {
      if (error === 0) return dataCallback(data); // TODO set help and mime?
      else return errorCallback(data);
    };
    // make the process a child process of current process
    this.adopt(procID, childReferenceID, callback);
    return;
  };

  this.pass = async (data_) => {
    data = data_;
    qrtzProcess.pass(data);
    return;
  };

  this.execute = async (data_) => { // TODO should not be exposed to non parent
    this.reset(data_);
    started = Date.now();
    qrtzStatement.execute(this);
    return;
  };

  this.done = async (data_) => {
    if (stopped) return;
    error = 0;
    data = data_;
    progress = 1;
    stopped = Date.now();
    qrtzProcess.done(data);
    return;
  };

  this.jump = async (delta, data_) => {
    if (stopped) return;
    error = 0;
    data = data_;
    progress = 1;
    stopped = Date.now();
    qrtzProcess.jump(delta, data);
    return;
  };

  this.next = async (data_) => {
    if (stopped) return;
    this.jump(1, data_);
  }

  this.kill = async () => {
    for (const childID in busyChildProcesses) {
      // TODO do not kill adopted childProcesses though!
      busyChildProcesses[childID].kill();
    }
  };

  this.fail = async (error_, data_) => {
    if (stopped) return;
    if (!error_) error_ = 1;
    if (typeof data_ === 'undefined') data_ = error_;
    if (data_ instanceof Error) data_ = this.getSessionID() === 1 ? data_.stack.toString() : data_.toString();
    error = error_;
    data = data_;
    stopped = Date.now();
    qrtzProcess.fail(error, data);
    return;
  };

  this.stop = async (error_, data_) => {
    if (error_ === 0) this.done(data_);
    else this.fail(error_, data_);
    return;
  };

  this.prog = async (step, steps) => {
    progress = typeof steps === 'undefined' || isNaN(steps) ? step : step / steps;
    qrtzProcess.prog(progress);
    return;
  };

  this.mime = async (mimetype) => qrtzProcess.mime(mimetype);
  this.help = async (help) => qrtzProcess.help(help);
  this.cach = async (...args) => qrtzProcess.cach(...args);

  this.rout = async (xpath, data, dataCallback, errorCallback, updateMeta = false) => {
    const sessionID = this.getSessionID();
    const request = {url: xpath, sessionID, data, hideInLogs: true};
    rout(request, result => {
      if (result.id === 'id') { // if result is a process that has to be awaited, then read that process until finished
        const callback = (error, data) => {
          if (error === 0) return dataCallback(data); // TODO set help and mime?
          else return errorCallback(data);
        };
        const processID = result.data;
        // make the process a child process of current process
        this.adopt(processID, 0, callback); // use a 0 childReferenceID
      } else {
        if (updateMeta) {
          this.help(result.help);
          this.cach(result.cach);
          this.mime(result.type);
        }
        const data = result.hasOwnProperty('data') ? result.data : null;
        if (!result.hasOwnProperty('error') || result.error === 0) dataCallback(data);
        else return errorCallback(data);
      }
    });
    return;
  };

  this.setProgChildStep = async (childProgress) => {
    qrtzProcess.setProgChildStep(childProgress);
    return;
  };
  this.setAutoProg = async (enabled) => {
    qrtzProcess.setAutoProg(enabled);
    return;
  };

  this.setTimeOut = async (timeOut) => {
    qrtzProcess.setTimeOut(timeOut);
    return;
  };

  this.setChildrenTimeOut = async (timeOut) => {
    for (const childID in busyChildProcesses) {
      busyChildProcesses[childID].setTimeOut(timeOut);
    }
    return;
  };

  this.getMime = () => qrtzProcess.getMime();
  this.getHelp = () => qrtzProcess.getHelp();
  this.getProgress = () => progress;
  this.getStep = () => step;
  this.getSteps = () => qrtzProcess.getStepCount();
  this.getName = () => qrtzProcess.getName() + ':' + (step + 1);
  this.getProcessID = () => qrtzProcess.getProcessID() + '.' + step;
  this.getData = () => data;
  this.getError = () => error;
  this.getProcessData = () => qrtzProcess.getData();
  this.getTimeOut = () => qrtzProcess.getTimeOut();
  this.getCommand = index => qrtzProcess.getCommand(index);
  this.getRecipe = () => qrtzProcess.getRecipe();
  this.getVars = () => qrtzProcess.getVars();
  this.getParentVars = () => qrtzProcess.getParentVars();
  this.getSessionID = () => qrtzProcess.getSessionID();
  this.hasStarted = () => !!started;
  this.hasStopped = () => !!stopped;
  this.hasParentStarted = () => !!qrtzProcess.getParentStarted();
  this.hasParentStopped = () => !!qrtzProcess.getParentStopped();
  this.getInfo = (sessionID_) => {
    return this.getSessionID() !== sessionID_ && sessionID_ !== 1
      ? get403(this.getProcessID())
      : {id: this.getProcessID(), error, data, started, stopped, progress};
  };

  this.addDebug = async (result, sessionID, prefix) => {
    prefix += '.' + step;
    result[prefix] = {...this.getInfo(sessionID), labels: {}, qrtz: this.toString()};
    for (const subID in busyChildProcesses) {
      busyChildProcesses[subID].addDebug(result, sessionID, prefix + '.' + subID);
    }
    for (const subID in finishedChildProcesses) {
      finishedChildProcesses[subID].addDebug(result, sessionID, prefix + '.' + subID);
    }
    return;
  };

  const logger = logType => async function (...messages) {
    const recipe = this.getRecipe();
    const id = typeof recipe === 'object' && recipe !== null
      ? recipe.id
      : undefined;
    global.hybrixd.logger.apply(global.hybrixd.logger, [[logType, id], ...messages]);
  }.bind(this);

  this.warn = logger('error');
  this.logs = logger('run');
  this.info = logger('info');

  this.func = (command, data, parentProcID) => {
    const head = command[0];
    const moduleID = this.getRecipe().module;
    if (!modules.module.hasOwnProperty(moduleID)) {
      this.fail(`Unknown module '${moduleID}'`);
    } else if (!modules.module[moduleID].main.hasOwnProperty(head)) {
      this.fail(`Unknown function '${head}' for module '${moduleID}'`);
    } else {
      const pass = data => this.pass(data);
      const done = data => this.next(data);
      const stop = (err, data) => { if (err !== 0) { this.fail(err, data); } else { this.next(data); } };
      const fail = (err, data) => this.fail(err, data);
      const prog = (step, steps) => this.prog(step, steps);
      const cach = cach_ => this.cach(cach_);
      const mime = mimetype => this.mime(mimetype);
      const peek = (key, fallback) => {
        const result = this.peek(key);
        return result.e || typeof result.v === 'undefined'
          ? fallback
          : result.v;
      };
      const poke = async (key, value) => this.poke(key, value);
      const help = async (helpMessage) => this.help(helpMessage);
      const fork = async (command, data) => {
        if (typeof command !== 'string') return this.fail(1, 'fork: Expected string path');
        else return this.fork(command.split('/'), data, false);
      };
      const read = async (procID) => {
        return new Promise((resolve) => {
          function warning(message) {
            global.hybrixd.logger(['error', 'process'], `func executed read: ${message}`);
            resolve(null);
          }
          if (processExists(procID, 1)) {
            let readTimeout = false;
            const readInterval = setInterval( () => {
              procInfo = getInfo(procID, 1);
              if (!readTimeout) readTimeout = setTimeout( () => {
                resolve(procInfo);
                clearInterval(readInterval);
              }, procInfo.timeout);
              if (procInfo.stopped > 0) {
                resolve(procInfo);
                clearInterval(readInterval);
                clearTimeout(readTimeout);
              }
            }, 100);
          } else {
            warning(`Process does not exist within session (${this.sessionID})`);
            resolve(null);
          }
        });
      }
      const call = async (command, data) => {
        if (typeof command !== 'string') return this.fail(1, 'call: expected string path');
        else {
          const procID = await fork(command, data);
          return await read(procID);
        }
      }
      const rout = async (path, data, dataCallback, errorCallback) => this.rout(path, data, dataCallback, errorCallback);

      const getMime = () => this.getMime();
      const getHelp = () => this.getHelp();
      const getProgress = () => this.getProgress();

      const host = async (fileName) => {
        const fileNameSplitByDot = fileName.split('.');
        const extension = fileNameSplitByDot[fileNameSplitByDot.length - 1];
        const mimeType = mimeTypesByExtension.hasOwnProperty(extension) ? mimeTypesByExtension[extension] : 'text/html';
        if (extensionsWithCache.includes(extension)) this.cach('max-age=604800');
        this.mime('file:' + mimeType);
        this.done(fileName.split('?')[0]);
      };

      const sessionID = this.getSessionID();

      try {
        data = modules.module[moduleID].main[head]({
          pass,
          done,
          stop,
          fail,
          peek,
          poke,
          prog,
          cach,
          mime,
          help,
          rout,
          fork, // TODO: fork with callback?
          call,
          read,
          host,
          getMime,
          getHelp,
          getProgress,
          sessionID,
          logs: this.logs,
          info: this.info,
          warn: this.warn,
          command: JSON.parse(JSON.stringify(command))
        }, data);
      } catch (error) {
        const internalErrorMessage = `Javascript module failure for ${moduleID}.${head} : ${error instanceof Error ? error.stack : error}`;
        global.hybrixd.logger(['error', 'process'], internalErrorMessage);
        const externalErrorMessage = this.getSessionID() === 1 // root session gets full details on error
          ? internalErrorMessage
          : 'Javascript module failure';
        this.fail(1, externalErrorMessage);
      }
    }
  }
}

/**
 *
 */
async function purgeProcesses () {
  const procPurgeTime = (conf.get('scheduler.ProcPurgeTime') || PROCESS_PURGE_TIME) * 1000;
  const now = Date.now();
  for (const processID in busyProcesses) {
    const process = busyProcesses[processID];
    const timeOut = process.getTimeOut();
    if (timeOut >= 0 && process.getStarted() < now - timeOut) { // set error when process has timed out on its short-term action
      global.hybrixd.logger(['error', 'process'], `Busy process ${processID} timed out after ${timeOut}ms. -> ${JSON.stringify(process.getName())}`);
      process.fail(`Busy process timed out after ${timeOut}ms. -> ${JSON.stringify(process.getName())}`);
    }
  }
  for (const processID in finishedProcesses) {
    const process = finishedProcesses[processID];
    if (process.getStopped() < now - procPurgeTime) delete finishedProcesses[processID];
  }
  return;
}

/**
 * @param maxUsage
 * @param maxParallelProcesses
 */
async function updateProcesses (maxUsage, maxParallelProcesses) {
  let now = Date.now();
  const end = now + maxUsage; // end of the allowed timeframe
  let schedulerParallelProcesses = Object.keys(queuedProcesses).length;
  while (now <= end && schedulerParallelProcesses > 0) { // run through all processes for a maximum timeframe if there are active processes
    const currentProcesses = Object.values(queuedProcesses).slice(0);
    schedulerParallelProcesses = currentProcesses.length;
    for (let i = 0; i < currentProcesses.length && i < maxParallelProcesses; ++i) {
      currentProcesses[i].update(now);
    }
    now = Date.now();
  }
}

/**
 *
 */
async function killAll () {
  for (const processID in queuedProcesses) {
    const process = queuedProcesses[processID];
    process.kill();
  }
  for (const processID in busyProcesses) {
    const process = busyProcesses[processID];
    process.kill();
  }
  return;
}

/**
 * @param properties
 * @param recipe
 * @param command
 */
function getQrtzFunction (properties, recipe, command) {
  if (properties.hasOwnProperty('function') && properties.function instanceof QrtzFunction) {
    return properties.function;
  } else if (properties.hasOwnProperty('steps') && properties.steps instanceof Array) {
    return new QrtzFunction(properties.steps);
  } else if (recipe.hasOwnProperty('quartz')) {
    if (command.length > 0) {
      for (const functionSignature in recipe.quartz) {
        const functionSignatureSplit = functionSignature.split('/');
        const functionName = functionSignatureSplit[0];
        if (functionName === command[0]) {
          const stepsOrFunction = recipe.quartz[functionSignature];
          if (stepsOrFunction instanceof Array) {
            return new QrtzFunction(stepsOrFunction, functionSignatureSplit);
          } else {
            return stepsOrFunction;
          }
        }
      }
    }
    if (recipe.quartz.hasOwnProperty('_root')) {
      const stepsOrFunction = recipe.quartz._root;
      if (stepsOrFunction instanceof Array) {
        return new QrtzFunction(stepsOrFunction);
      } else {
        return stepsOrFunction;
      }
    } else {
      global.hybrixd.logger(['error', 'process'], 'Failed to create process', command);
      return new QrtzFunction([['fail', 'Failed to create process']]);
    }
  } else {
    global.hybrixd.logger(['error', 'process'], 'Failed to create process', command);
    return new QrtzFunction([['fail', 'Failed to create process']]);
  }
}

/**
 * @param properties
 */
let lastProcessSubIDInThisTimeWindow = 0;
function getProcessID (properties) {
  if (properties.parent) return `${properties.parent.getProcessID()}.${properties.childID}`;
  else {
    const now = Date.now();
    if (lastProcessSubIDInThisTimeWindow < 1000) {
      lastProcessSubIDInThisTimeWindow++;
    } else lastProcessSubIDInThisTimeWindow = 0;
    const processSubID = lastProcessSubIDInThisTimeWindow;
    const suffix = `00${processSubID}`.slice(-3);
    return `${now}${suffix}`;
  }
}

/**
 * @param properties
 */
  // [!] do not expose outside this file
function QrtzProcess (properties) {
  // properties
  // - sessionID,
  // - data,
  // - command,
  // - [recipe,]
  // - [function],[steps]
  // - parent : the parent ProcessStep
  // - childID : the actual child id
  // - childReferenceID : an id used for reference by the parent ProcessStep

  // TODO wchan
  // - mime
  //      offset: process.offset,
  //    length: process.length,
  //    hash: process.hash,

  const parent = properties.parent;
  const postProcessCallback = properties.postProcessCallback;

  const childID = properties.childID;
  const childReferenceID = properties.childReferenceID;

  const processID = getProcessID(properties);

  let data = properties.hasOwnProperty('data') ? properties.data : null;
  let error = 0;

  const command = properties.command || {};

  const recipe = properties.recipe || {};
  const qrtzFunction = getQrtzFunction(properties, recipe, command);

  let busy = false;
  let step = null;

  let progress = null; // 0 is started, 1 = completed
  let autoprog = true; // whether to use step/stepCount as progress
  let timeout = properties.timeout || DEFAULT_PROCESS_TIMEOUT;

  const started = Date.now();
  let stopped = null;
  const sessionID = properties.sessionID || 0;

  let mime;
  let help;
  let cacheControlParameters = [];

  const adopters = []; // this process can become a child process for another process, as happens with the qrtz rout command

  let hook = null; // determines on failure behaviour (for process errors and API queue time outs)
  const timeHook = null; // determines on failure behaviour (for process time outs)
  const vars = properties.vars || {}; // variables used by quartz.poke and quartz.peek

  if (qrtzFunction instanceof QrtzFunction) {
    qrtzFunction.setNamedCommandVars(command, vars);
  }

  const processSteps = qrtzFunction instanceof QrtzFunction
    ? qrtzFunction.getStatements().map((qrtzStatement, step) => new QrtzProcessStep(qrtzStatement, this, step))
    : null;

  const reportStopToParents = (error, data) => {
    if (parent) {
      parent.childReportsStop(error, data, childID, childReferenceID, postProcessCallback);
    }
    for (const adopter of adopters) {
      const parent = adopter.parent;
      const childID = adopter.childID;
      const childReferenceID = adopter.childReferenceID;
      const postProcessCallback = adopter.postProcessCallback;
      parent.childReportsStop(error, data, childID, childReferenceID, postProcessCallback);
    }
  };

  this.adopt = properties => { // this process is being adopted as a child by another process
    adopters.push(properties);
  };

  this.done = data_ => {
    if (stopped) return;
    error = 0;
    progress = 1;
    data = data_;
    this.kill();
  };

  this.fail = (error_, errorMessage) => {
    if (stopped) return;
    if (typeof errorMessage === 'undefined') {
      errorMessage = error_;
      error_ = 1;
    }

    if (hook) {
      if (hook.hasOwnProperty('jump') && !isNaN(hook.jump)) {
        const ydata = hook.hasOwnProperty('data') ? hook.data : data;
        const jump = hook.jump - step;
        hook = null;
        return this.jump(jump, ydata);
      } else if (hook.hasOwnProperty('data')) {
        if (hook.error === 0) {
          const ydata = hook.data;
          hook = null;
          return this.done(ydata);
        } else {
          error = hook.error;
          data = hook.data;
          hook = null;
          return this.kill();
        }
      }
    }
    data = errorMessage;
    error = error_;
    this.kill();
  };

  this.kill = () => {
    stopped = Date.now();
    busy = false;
    finishedProcesses[processID] = this;
    delete busyProcesses[processID];
    delete queuedProcesses[processID];
    reportStopToParents(error, data);
    for (const processStep of processSteps) {
      processStep.kill();
    }
  };

  this.jump = (delta, data_) => {
    if (stopped) return;
    data = data_;
    if (delta + step === qrtzFunction.getStepCount()) this.done(data);
    else if (delta + step === 0) this.fail('jump: Illegal zero jump, or process halted.');
    else if (delta + step < 0) this.fail('jump: Illegal backwards out of bounds jump.');
    else if (delta + step > qrtzFunction.getStepCount()) this.fail('jump: Illegal forwards out of bounds jump.');
    else {
      step += delta;
      busy = false;
      delete busyProcesses[processID];
      queuedProcesses[processID] = this;
    }
  };

  this.pass = data_ => {
    data = data_;
  };

  this.read = data_ => {
    if (stopped) return;
    if (busy === false) return this.done(data);
  };

  this.update = now => {
    if (timeout && started + timeout < now) {
      const process = queuedProcesses[processID];
      const processName = process.hasOwnProperty('getName') && typeof process.getName === 'function' ? JSON.stringify(process.getName()) : '[unknown]'
      global.hybrixd.logger(['error', 'process'], `Queued process ${processID} timed out after ${timeout}ms. / ${processName}`);
      this.fail(`Queued process timed out after ${timeout}ms. / ${processName}`);
      return;
    }
    delete queuedProcesses[processID];

    if (stopped) {
      finishedProcesses[processID] = this;
      delete busyProcesses[processID];
      return;
    }

    if (busy) return;
    busyProcesses[processID] = this;
    busy = true;

    if (autoprog) {
      progress = step / qrtzFunction.getStepCount();
      if (parent) {
        parent.setProgChildStep(progress);
      }
    }
    if (step === processSteps.length) { this.done(data); return; }
    if (step > processSteps.length) { this.fail('Illegal out of bounds step.'); return; }
    if (step < 0) { this.fail('Illegal negative step.'); return; }
    if (isNaN(step)) { this.fail('Illegal non numeric step.'); return; }
    processSteps[step].execute(data);
  };

  this.hook = (step, jumpOrErrOrData, error) => {
    if (typeof jumpOrErrOrData === 'undefined' && typeof error === 'undefined') hook = null; // reset hook
    else if (isNaN(jumpOrErrOrData)) hook = {data: jumpOrErrOrData, error};
    else if (Number(jumpOrErrOrData) === 0) this.fail('hook: Illegal zero jump, or process forcefully halted.');
    else if (Number(jumpOrErrOrData) + step < 0) this.fail('hook: Illegal backward out of bounds jump.');
    else if (Number(jumpOrErrOrData) + step > qrtzFunction.getStepCount()) this.fail('hook: Illegal forward out of bounds jump.');
    else hook = {jump: Number(jumpOrErrOrData) + step, error: 0};
  };

  this.mime = mime_ => { mime = mime_; };
  this.cach = (...args) => { cacheControlParameters = args; };
  this.prog = progress_ => { progress = progress_; autoprog = false; };
  this.help = help_ => { help = help_; };

  this.setProgChildStep = childProgress => {
    if (autoprog) {
      progress = (step / qrtzFunction.getStepCount()) + (childProgress / qrtzFunction.getStepCount());
      if (parent) {
        parent.setProgChildStep(progress);
      }
    }
  };

  this.setAutoProg = enabled => { autoprog = enabled; progress = step / qrtzFunction.getStepCount(); };

  this.setTimeOut = milliseconds => {
    if (milliseconds === -1 || milliseconds > DEFAULT_PROCESS_TIMEMAX) milliseconds = DEFAULT_PROCESS_TIMEMAX;
    if (milliseconds > timeout) {
      timeout = milliseconds;
      for (const processStep of processSteps) {
        processStep.setChildrenTimeOut(milliseconds);
      }
      if (parent) parent.setTimeOut(milliseconds);
    }
  };

  this.getStepCount = () => { return qrtzFunction.getStepCount(); }

  this.getInfo = (sessionID_) => {
    return sessionID !== sessionID_ && sessionID_ !== 1
      ? get403(processID)
      : {id: processID, error, data, mime, started, stopped, progress, help, command, timeout, cach: cacheControlParameters};
  };
  this.getFollowUp = (sessionID_) => {
    if (sessionID !== sessionID_ && sessionID_ !== 1) {
      return get403(processID);
    } else {
      return stopped
        ? this.getInfo(sessionID_)
        : {id: 'id', error, data: processID, mime, started, stopped, progress, help, command};
    }
  };
  this.getName = () => {
    let type;
    if (recipe.hasOwnProperty('symbol')) type = 'asset';
    else if (recipe.hasOwnProperty('engine')) type = 'engine';
    else if (recipe.hasOwnProperty('source')) type = 'source';
    else type = 'script';
    const id = recipe.symbol || recipe.id || createHash('sha256').update(JSON.stringify(recipe)).digest('hex');
    return '/' + type + '/' + id + '/' + qrtzFunction.getName();
  };
  this.getMime = () => mime;
  this.getHelp = () => help;
  this.getProgress = () => progress;

  this.getProcessID = () => processID;
  this.getSessionID = () => sessionID;
  this.getStarted = () => started;
  this.getStopped = () => stopped;

  this.getCommand = index => typeof index === 'undefined' ? command : command[index]; // TODO stringify/parse to ugly copy

  this.getRecipe = () => recipe;
  this.getVars = () => vars;
  this.getParentVars = () => parent instanceof QrtzProcessStep ? parent.getVars() : {};
  this.getParentStarted = () => parent instanceof QrtzProcessStep ? parent.hasStarted() : false;
  this.getParentStopped = () => parent instanceof QrtzProcessStep ? parent.hasStopped() : true;
  this.getTimeOut = () => timeout;
  this.getData = () => data;
  this.getError = () => error;

  this.addDebug = (result, sessionID, prefix) => {
    prefix = prefix || this.getProcessID();
    result[prefix] = {
      ...this.getInfo(sessionID),
      labels: qrtzFunction.getLabels(),
      qrtz: '/' + command.join('/'),
      name: this.getName()
    };
    for (let step = 0; step < processSteps.length; ++step) {
      const processStep = processSteps[step];
      processStep.addDebug(result, sessionID, prefix);
    }
    return result;
  };

  let servResult;
  if (!(qrtzFunction instanceof QrtzFunction)) {
    this.fail('Missing Qrtz function');
  } else if ((servResult = qrtzFunction.getSyncFunctionName())) {
    // handle direct hosting operations serv and host
    const [head, parameter] = servResult;
    queuedProcesses[processID] = this;
    progress = 0;
    step = 0;
    switch (head) {
      case 'serv':
        // use proc.pass(data) in synchronous functions inside modules to pass data instantaneously
        processSteps[0].func([parameter].concat(command), data);
        progress = 1;
        stopped = Date.now();
        break;
      case 'host':
        const fileNameSplitByDot = parameter.split('.');
        const extension = fileNameSplitByDot[fileNameSplitByDot.length - 1];
        const mimeType = mimeTypesByExtension.hasOwnProperty(extension) ? mimeTypesByExtension[extension] : 'text/html';
        if (extensionsWithCache.includes(extension)) this.cach('max-age=604800');
        this.mime('file:' + mimeType);
        data = parameter;
        progress = 1;
        stopped = Date.now();        
        break;
      default:
        error = 500;
        data = `Illegal serv/host method '${head}'`;
        stopped = Date.now();
    }
  } else if (qrtzFunction.getStepCount() === 0) this.done();
  else {
    queuedProcesses[processID] = this;
    step = 0;
    progress = 0;
  }
}

/**
 * @param properties
 */
function create (properties) {
  const process = new QrtzProcess(properties);
  return process.getFollowUp(properties.sessionID);
}

/**
 * @param processID
 * @param sessionID
 */
function getInfo (processID, sessionID) {
  const process = getProcess(processID);
  return process
    ? process.getInfo(sessionID)
    : get404(processID);
}

/**
 * @param processID
 * @param sessionID
 */
function getFollowUp (processID, sessionID) {
  const process = getProcess(processID);
  return process
    ? process.getFollowUp(sessionID)
    : get404(processID);
}

/**
 * @param processID
 * @param sessionID
 */
function getDebug (processID, sessionID) {
  const process = getProcess(processID);
  if (process) {
    const info = process.getInfo(sessionID);
    const data = process.addDebug({}, sessionID);
    data[processID].data = info.data;
    info.error = 0;
    info.data = data;
    return info;
  } else {
    return get404(processID);
  }
}

/**
 * @param processID
 */
function processExists (processID, sessionID) {
  if (sessionID === 0 || typeof sessionID === 'undefined') return [];
  return queuedProcesses.hasOwnProperty(processID) ||
    busyProcesses.hasOwnProperty(processID) ||
    finishedProcesses.hasOwnProperty(processID);
}

/**
 * @param processes
 * @param sessionID
 */
function getSessionProcessKeys (processes, sessionID) {
  if (sessionID === 0 || typeof sessionID === 'undefined') return [];
  else if (sessionID === 1) return Object.keys(processes);
  else return Object.keys(processes).filter(processID => processes[processID].getSessionID() === sessionID);
}

function getBusyProcessList (sessionID) {
  return getSessionProcessKeys(busyProcesses, sessionID);
}

function getQueuedProcessList (sessionID) {
  return getSessionProcessKeys(queuedProcesses, sessionID);
}

function getFinishedProcessList (sessionID) {
  return getSessionProcessKeys(finishedProcesses, sessionID);
}

/**
 * @param sessionID
 */
function getProcessList (sessionID) {
  const list = getQueuedProcessList(sessionID)
    .concat(getBusyProcessList(sessionID))
    .concat(getFinishedProcessList(sessionID));
  return list;
}

/**
 * @param processID
 * @param data
 */
function done (processID, data) {
  const process = getProcess(processID);
  if (!process) return false;
  process.done(data);
  return true;
}

/**
 * @param processID
 * @param data
 */
function fail (processID, data) {
  const process = getProcess(processID);
  if (!process) return false;
  process.fail(data);
  return true;
}

/**
 * @param processID
 * @param data
 */
function kill (processID, data) {
  const process = getProcess(processID);
  if (!process) return false;
  process.kill(data);
  return true;
}

/**
 * @param processID
 * @param progress
 */
function prog (processID, progress) {
  const process = getProcess(processID);
  if (!process) return false;
  process.prog(progress);
  return true;
}

// return a function that takes a sequential callback array
/**
 * @param processId
 * @param data
 */
function sequentialStop (processId, data) {
  return async (cbArr) => {
    done(processId, data);
    sequential.next(cbArr);
  };
}

exports.sequentialStop = sequentialStop;

exports.processExists = processExists;
exports.getProcessList = getProcessList;
exports.getBusyProcessList = getBusyProcessList;
exports.getQueuedProcessList = getQueuedProcessList;
exports.getFinishedProcessList = getFinishedProcessList;

exports.create = create;

exports.done = done;
exports.fail = fail;

exports.failure = fail;
exports.prog = prog;
exports.kill = kill;
exports.getDebug = getDebug;
exports.getInfo = getInfo;
exports.getFollowUp = getFollowUp;

exports.stopAll = killAll;

exports.updateProcesses = updateProcesses;
exports.purgeProcesses = purgeProcesses;
