const sha256 = require('../../common/node_modules/js-sha256');
const storageLib = require('../../common/storage-fileBased');
const {safeDecompress} = require('../../common/crypto/urlbase64');
const DEFAULT_SALT = require('../../lib/scheduler/methods/decr.js').DEFAULT_SALT;
const CLAIMLIST_MAX_SIZE = 1000;
const TXID_REGEX = '^[:.,a-zA-Z0-9]+$'

// determine the transaction status
async function txStatus (proc,data) {
  const timeout = data.status === 'timeout' ? 1 : 0;
  const rejected = data.status === 'rejected' ? 1 : 0;
  const done = data.progress === 1 ? 1 : 0;
  return proc.done({timeout,rejected,done});
}

// create claim hash for manual swap transactions
function claimHash(input) {
  return sha256.sha256(`${global.hybrixd.node.secretKey}${input}`)
}

async function claimCreateHash (proc) {
  const input = proc.command[1];
  if (input) proc.done(claimHash(input));
  else proc.fail(1, 'Cannot create claim hash without input');
  return;
}

// read back deal history
async function readHistory (proc) {
  const timeOffset = proc.command[1];
  const timeLimit = proc.command[2];
  if (isNaN(timeOffset) || isNaN(timeLimit)) {
    proc.fail(1, 'Expecting numeric timeOffset and timeLimit!');
  } else {
    const completeList = proc.peek(`local::completeList`,[]);
    let result = [];
    for (const entry of completeList) {
      const loadSync = new Promise((resolve, reject) => {
        const nullReturn = () => { resolve(null); }
        storageLib.load({key: entry, type: 'string'}, resolve, nullReturn);
      });
      const nullReturn = () => { resolve(null); }
      try {
        const loadData = JSON.parse(await loadSync);
        // decrypt loaded data
        if (typeof loadData === 'object' && loadData !== null && loadData.hasOwnProperty('timestamp')) {
          const timeNow = Date.now() / 1000;
          const timeUntil = timeNow - timeOffset;
          const timeFrom = timeUntil - timeLimit;
          if (loadData.timestamp >= timeFrom && loadData.timestamp < timeUntil) result.push(loadData);
        }
      } catch (e) {
        // DEBUG: global.hybrixd.logger(['error', 'engine-deal'], `readHistory: could not read or parse data of "${entry}"`);
      }
      /* DEPRECATED
      const keys = {publicKey: global.hybrixd.node.publicKey, secretKey: global.hybrixd.node.secretKey};
      let decrData = {};
      try {
        const user_keys = {boxPk: nacl.from_hex(keys.publicKey), boxSk: nacl.from_hex(keys.secretKey.substr(0, 64))}; // cut off pubkey for boxSk!
        const nonce_salt = nacl.from_hex(DEFAULT_SALT);
        const crypt_hex = nacl.from_hex(safeDecompress(loadData));
        const crypt_bin = nacl.crypto_box_open(crypt_hex, nonce_salt, user_keys.boxPk, user_keys.boxSk); // use nacl to create a crypto box containing the data
        decrData = JSON.parse(nacl.decode_utf8(crypt_bin));
      } catch (e) {
        global.hybrixd.logger(['error', 'engine-deal'], 'readHistory: could not decrypt or parse data! -> '+e);
      }
      // TODO: jpar then test timestamp
      if (decrData.hasOwnProperty('timestamp')) {
        const timeNow = Date.now() / 1000;
        const timeUntil = timeNow - timeOffset;
        const timeFrom = timeUntil - timeLimit;
        if (decrData.timestamp >= timeFrom && decrData.timestamp < timeUntil) result.push(decrData);
      }
      */
    }
    proc.done(result);
  }
}

// push and claim for swap transactions
function procSuccess (procResult) {
  return procResult.hasOwnProperty('data') && procResult.data !== null;
}

async function pushAndClaim (proc, data) {
  const proposalID = proc.command[1];
  const pushSymbol = proc.command[2];
  const pushRawTx = proc.command.slice(3).join('/');
  // push rawTx
  let procResult = await proc.call(`${pushSymbol}::push/${pushRawTx}`);
  if (!procSuccess(procResult)) {
    proc.warn(`Could not push remittance for ${pushSymbol} -> ${JSON.stringify(procResult)}`);
    return proc.fail(1, `Could not push remittance for ${pushSymbol}`);
  }
  const txid = procResult.data;
  claimTransaction(proc, txid);
}

async function claimTransaction (proc, txid) {
  const proposalID = proc.command[1];
  // check if txid is on claim list
  const txidRegex = new RegExp(TXID_REGEX);
  if (txidRegex.test(txid)) {
    // check if txid is on claim list
    const claimList = proc.peek('local::claimList', []);
    if (claimList.includes(`${proposalID}:${txid}`)) {
      const message = `Claim for ${proposalID} was already submitted, txid ${txid}`;
      proc.warn(message);
      return proc.done(message);
    }
    if (claimList.length > CLAIMLIST_MAX_SIZE) claimList.shift();
    claimList.push(`${proposalID}:${txid}`);
    proc.poke('local::claimList', claimList);
    // register txid in swap data object
    procResult = await proc.call(`getProposal/${proposalID}`);
    if (!procSuccess(procResult)) return proc.fail(1, `Could not get proposal ${proposalID}`);
    let proposalObj = procResult.data;
    if (proposalObj.hasOwnProperty('deal') && proposalObj.deal.hasOwnProperty('ask') && proposalObj.deal.hasOwnProperty('bid')) {
      const virtualPrefix = proc.peek('virtualPrefix');
      let claimType = 'ask';
      // determine if claim is virtual/real, ask/bid
      if (typeof proposalObj.deal.ask.symbol === 'string'
          && proposalObj.deal.ask.symbol.includes(virtualPrefix)
          && txid !== claimHash(proposalObj.deal.id)) return proc.fail('Invalid claim for ${proposalID}, txid ${txid}');
      else if (proposalObj.deal.ask.txid
               && typeof proposalObj.deal.bid.symbol === 'string'
               && proposalObj.deal.bid.symbol.includes(virtualPrefix)
               && !proposalObj.deal.bid.txid) claimType = 'bid';
      // finalize claim
      if (proposalObj.deal[claimType].txid) {
        proc.warn(`Transaction ID for ${proposalID} already submitted`);
      } else {
        proposalObj.deal[claimType].txid = txid;
        proc.call('saveProposal', proposalObj);
        const accountID = proposalObj.accountID;
        const lockSymbol = proposalObj.deal[claimType].symbol;
        if (accountID && lockSymbol) proc.call(`setAccountLock/${accountID}/${lockSymbol}`);
      }
      return proc.done(txid);
    } else return proc.fail(1, `ProposalID {proposalID} object corrupted`);
  } else {
    proc.warn(`Invalid remittance transaction! -> ${JSON.stringify(txid)}`);
    return proc.fail(1, 'Remittance transaction is invalid');
  }
}

// exports
exports.txStatus = txStatus;
exports.claimCreateHash = claimCreateHash;
exports.pushAndClaim = pushAndClaim;
exports.claimTransaction = claimTransaction;
exports.readHistory = readHistory;
